/** mbed library for Dialog Semiconductor DA14580 Bluetooth Smart chip
 * by Kazuki Yamamoto, or _K4ZUKI_
 **/

#ifndef __DA14580_H__
#define __DA14580_H__

#include "mbed.h"
#include "W25X40BV.h"

#define     LOADER_FILE         "/local/loader.bin"
#define     TARGET_FILE         "/local/target.bin"
#define     _RESET   0
#define     _BOOT   1
#define     _TIMEOUT 10000

/** mbed library for Dialog Semiconductor DA14580 Bluetooth Smart chip
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "DA14580.h"
 *
 * DA14580 BLE(P0_18, P0_19, P0_1);
 * Serial pc(USBTX, USBRX);
 * LocalFileSystem local( "local" );
 *
 * #undef      LOADER_FILE
 * #define     LOADER_FILE         "/local/loader_bin"
 *
 * #undef      TARGET_FILE
 * #define     TARGET_FILE         "/local/target_bin"
 *
 * int main()
 * {
 *    int result=0;
 *    pc.baud(115200);
 *
 *   wait_ms(1);
 *   result = BLE.load();
 *   pc.printf("Result = %d \n\r",&result);
 * }
 * @endcode
 */
class DA14580
{
public:
    /** \enum XMODEM_CONST
        \brief Constant number used in XMODEM
        @param SOH = 0x01
        @param STX = 0x02
        @param EOT = 0x04
        @param ACK = 0x06
        @param DLE = 0x10
        @param NAK = 0x15
        @param CAN = 0x18
    */
    enum XMODEM_CONST {
        SOH = (0x01),
        STX = (0x02),
        EOT = (0x04),
        ACK = (0x06),
        DLE = (0x10),
        NAK = (0x15),
        CAN = (0x18),
    };

    /** \enum DA14580_STATUS
        \brief Status flags to show write operation is succeeded 
        @param SUCCESS =               '0'
        @param E_NOT_CONNECTED =       '1'
        @param E_FILE_NOT_FOUND =      '2'
        @param E_TIMEOUT_STX =         '3'
        @param E_ACK_NOT_RETURNED =    '4'
        @param E_CRC_MISMATCH =        '5'
    */
    enum DA14580_STATUS {
//        SUCCESS = '0',
//        E_NOT_CONNECTED = '1',
//        E_FILE_NOT_FOUND = '2',
//        E_TIMEOUT_STX = '3',
//        E_ACK_NOT_RETURNED = '4',
//        E_CRC_MISMATCH = '5'
        SUCCESS = 0,
        E_NOT_CONNECTED = 1,
        E_FILE_NOT_FOUND = 2,
        E_TIMEOUT_STX = 3,
        E_ACK_NOT_RETURNED = 4,
        E_CRC_MISMATCH = 5
    };

    /** Constructor
    * @param TX UART data output from mbed
    * @param RX UART data input from device
    * @param RESET LOW-active RESET signal
    */
    DA14580( PinName TX, PinName RX, PinName RESET );

    /** Constructor
    * @param &ble pointer to RawSerial instance
    * @param RESET LOW-active RESET signal
    */
    DA14580( RawSerial &ble, PinName RESET );

    /** Destructor
    */
    ~DA14580();

    /** Initialise internal variables
    */
    void init();

    /** copy application executable into external SPI flash memory
    */
    void copy_to_flash(W25X40BV* flash);

    /** Load bootloader executable into device through UART
    * the target binary should be defined by LOADER_FILE
    */
    int load(char *filename = LOADER_FILE);

    /** Returns file size: copied from ika_shouyu_poppoyaki
    * @param *fp File pointer
    */
    int file_size( FILE *fp );

    RawSerial _ble;

private:
    uint8_t _recieve;
    uint8_t _read;
    int _filesize;
    int _loadersize;
    int _timeout;
    int _status;
    FILE* _fp;
    int _crc;
    DigitalOut _reset;
};

#endif //__DA14580_H__
