/* mbed Keypad library, using user-defined interrupt callback
 * Copyright (c) 2012 Yoong Hor Meng
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE
 */

#ifndef KEYPAD_H
#define KEYPAD_H

#include "mbed.h"
#include "FPointer.h"

/**
 * An interrupt-based interface to 4x4 keypad.
 *
 * On each key pressed on a keypad, the index of the key is passed to a
 * user-defined function. User is free to define what to be done with the
 * input.
 *
 * This library makes use of
 * @see http://mbed.org/cookbook/FPointer by Andy Kirkham
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "keypad.h"
 *
 * // Define your own keypad values
 * char Keytable[] = { '1', '2', '3', 'A',
 *                     '4', '5', '6', 'B',
 *                     '7', '8', '9', 'C',
 *                     '*', '0', '#', 'D'
 *                   };
 *
 * uint32_t cbAfterInput(uint32_t index) {
 *     printf("Index:%d => Key:%c\n", key, Keytable[index]);
 *     return 0;
 * }
 *
 * int main() {
 *     Keypad keypad(p25, p26, p27, p28, p21, p22, p23, p24);
 *     keypad.CallAfterInput(&cbAfterInput);
 *     keypad.Start();
 *
 *     while (1) {
 *         wait_ms(100);
 *     }
 * }
 * @endcode
 */
class Keypad
{
public:
    /** Create a Keypad interface
     *
     *  @param row<3..0>     Row data lines
     *  @param col<3..0>     Column data lines
     *  @param debounce_ms   Debounce in ms (Default to 20ms)
     */
    Keypad(PinName row3, PinName row2, PinName row1, PinName row0,
           PinName col3, PinName col2, PinName col1, PinName col0,
           int debounce_ms = 20);

    /** Start the keypad interrupt routines
     */
    void Start(void);

    /** Stop the keypad interrupt routines
     */
    void Stop(void);

    /** Scan the keyboard for a debounced pressed key
     */
    int DebouncedScan(void);

    /** Scan the keyboard for a pressed key
     */
    int Scan(void);

    /** Scan the keyboard for multiple debounced pressed keys
     */
    int DebouncedScanMultiple(void);

    /** Scan the keyboard for multiple pressed keys
     */
    int ScanMultiple(void);

    /** User-defined function that to be called when a key is pressed
     *  @param fptr           A function pointer takes a uint32_t and
     *                        returns uint32_t
     */
    void CallAfterInput(uint32_t (*fptr)(uint32_t));

protected:
    static const int row_count = 4;
    static const int col_count = 4;

    InterruptIn      _row0;
    InterruptIn      _row1;
    InterruptIn      _row2;
    InterruptIn      _row3;
    InterruptIn      *_rows[row_count];
    BusInOut         _cols;  // BusOut doesn't support mode() yet; need open drain to prevent short circuits...
    int              _debounce;
    FPointer         _input; // Called after each input

    void _callback();
    void _setupFallTrigger(void);
};

#endif // KEYPAD_H
