// macros, defines, globals,and prototypes for the TTU_F031 library

#ifndef TTU_CSC1300_H
#define TTU_CSC1300_H

#include "mbed.h"

/*************************************************************************
**
** Uncomment the following line IF (AND ONLY IF) you have removed SB18
**   from the backside of the Nucleo-F031K6 to un-short pins PB_7 and
**   PA_5.
**
*************************************************************************/
// #define      USE_LED3


/*************************************************************************
 * 
 * DEFINES and MACROS
 * 
 **************************************************************************/
#define     FALSE           0
#define     TRUE            (~FALSE)

// some useful bitmasks
#define     BIT0            0x00000001
#define     BIT1            0x00000002
#define     BIT2            0x00000004
#define     BIT3            0x00000008
#define     BIT4            0x00000010
#define     BIT5            0x00000020
#define     BIT6            0x00000040
#define     BIT7            0x00000080
#define     BIT8            0x00000100
#define     BIT9            0x00000200
#define     BIT10           0x00000400
#define     BIT11           0x00000800
#define     BIT12           0x00001000
#define     BIT13           0x00002000
#define     BIT14           0x00004000
#define     BIT15           0x00008000
#define     BIT16           0x00010000
#define     BIT17           0x00020000
#define     BIT18           0x00040000
#define     BIT19           0x00080000
#define     BIT20           0x00100000
#define     BIT21           0x00200000
#define     BIT22           0x00400000
#define     BIT23           0x00800000
#define     BIT24           0x01000000
#define     BIT25           0x02000000
#define     BIT26           0x04000000
#define     BIT27           0x08000000
#define     BIT28           0x10000000
#define     BIT29           0x20000000
#define     BIT30           0x40000000
#define     BIT31           0x80000000

// bitmask for individual LED segments
#define     SEG_A           0xFE
#define     SEG_B           0xFD
#define     SEG_C           0xFB
#define     SEG_D           0xF7
#define     SEG_E           0xEF
#define     SEG_F           0xDF
#define     SEG_G           0xBF
#define     SEG_DP          0x7F

// Some useful seven-segment patterns for display HEX digits
//     NOTE: User can easily create additional ones for other letters/symbols
#define PATTERN_BLANK       0xFF
#define PATTERN_0           (SEG_A & SEG_B & SEG_C & SEG_D & SEG_E & SEG_F)
#define PATTERN_1           (SEG_B & SEG_C)
#define PATTERN_2           (SEG_A & SEG_B & SEG_D & SEG_E & SEG_G)
#define PATTERN_3           (SEG_A & SEG_B & SEG_C & SEG_D & SEG_G)
#define PATTERN_4           (SEG_B & SEG_C & SEG_F & SEG_G)
#define PATTERN_5           (SEG_A & SEG_C & SEG_D & SEG_F & SEG_G)
#define PATTERN_6           (SEG_A & SEG_C & SEG_D & SEG_E & SEG_F & SEG_G)
#define PATTERN_7           (SEG_A & SEG_B & SEG_C)
#define PATTERN_8           (SEG_A & SEG_B & SEG_C & SEG_D & SEG_E & SEG_F & SEG_G)
#define PATTERN_9           (SEG_A & SEG_B & SEG_C & SEG_D & SEG_F & SEG_G)
#define PATTERN_A           (SEG_A & SEG_B & SEG_C & SEG_E & SEG_F & SEG_G)
#define PATTERN_B           (SEG_C & SEG_D & SEG_E & SEG_F & SEG_G)
#define PATTERN_C           (SEG_A & SEG_D & SEG_E & SEG_F)
#define PATTERN_D           (SEG_B & SEG_C & SEG_D & SEG_E & SEG_G)
#define PATTERN_E           (SEG_A & SEG_D & SEG_E & SEG_F & SEG_G)
#define PATTERN_F           (SEG_A & SEG_E & SEG_F & SEG_G)

/*************************************************************************
 * 
 * HW to peripheral pin mappings  -- based on F031K6 using adapter
 *    inserted into Edubase-V2 target                          (July 2020)
 * 
**************************************************************************/
#define     SCL_PIN         PA_9
#define     SDA_PIN         PA_10
#define     LED0_PIN        PA_12
#define     LED1_PIN        PB_0
#define     LED3_PIN        PB_7
#define     PWM1_PIN        PB_1
#define     LED2_PIN        PF_0
#define     CS_SD_PIN       PF_1
#define     SPK_PIN         PA_8
#define     CS_LCD_PIN      PA_11
#define     MOSI_PIN        PB_5
#define     MISO_PIN        PB_4
#define     SCLK_PIN        PB_3
#define     LGT_SENS_PIN    PA_0
#define     POT_PIN         PA_1
#define     TEMP_SENS_PIN   PA_3
#define     PWM2_PIN        PA_4
#define     SW4_PIN         PA_5
#define     SW5_PIN         PA_6
#define     CS_7SEG_PIN     PA_7
#define     TX_PIN          PA_12
#define     RX_PIN          PA_15          

/*************************************************************************
 * 
 * Other useful defines
 * 
 **************************************************************************/
#define     I2C_SPEED_STD       100000
#define     I2C_SPEED_FAST      400000
#define     I2C_SPEED_FASTPLUS  1000000

#define     SPI_SPEED_100KBPS   100000
#define     SPI_SPEED_500KBPS   500000
#define     SPI_SPEED_1MBPS     1000000

/*************************************************************************
 *
 * USEFUL MACROS
 *
 *************************************************************************/

/*************************************************************************
 *
 * G L O B A L     V A R I A B L E S
 *
 *************************************************************************/
extern SPI         spi;
//extern I2C         i2c;
extern Serial      pc;

// create the LEDs
extern DigitalOut      led0;
extern DigitalOut      led1;
extern DigitalOut      led2;
/* only uncomment the following line if your Nucleo32 F031K6
** has removed the solder jumper SB18 on the underside
*/
#ifdef  USE_LED3
extern DigitalOut      led3;
#endif

// create the speaker
extern DigitalOut      speaker;

//create the chip-selects
extern DigitalOut      cs_sd;
extern DigitalOut      cs_lcd;
extern DigitalOut      cs_7seg;

// create the switches
extern DigitalIn       sw4;
extern DigitalIn       sw5;

// create the analog sensors
extern AnalogIn        lightSensor;
extern AnalogIn        pot;
extern AnalogIn        tempSensor;
 
/*************************************************************************
 *
 * P R O T O T Y P E S
 *
 *************************************************************************/
void init_spi(int i_spiBaud);
//void init_i2c(int i_i2cBaud);
void init_serial(int i_serialBaud);
void init_leds(void);
void init_all(void);

/* TEXTLCD
 * 
 * A TextLCD library targeted the F031K6 + Edubase-V2 combination
 * implmements a 16x2 HD44780 LCD module using a 4-bit inteface
 * connected via a 74HC595 using the chip's SPI
 *
 * This work is based upon the TextLCD mbed library by Simon Ford
 * found at https://os.mbed.com/users/simon/code/TextLCD/
 *
 * Copyright (c) 2007-2010, sford, http://mbed.org
 * Copyright (c) 2020, jwbruce   (jwbruce@tntech.edu)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
/**  A TextLCD interface for driving a HD44780 LCD character
 *   module connected via a 74HC595 using the F031K6 SPI peripheral.
 *   Specifically, the Nucleo32 F031K6 boards plugged into the Edubase-V2
 *   via the adapter used at Tennessee Tech.
 *
 * Supports only the 16x2 display on the EduBase-V2 board
 *
 * @code
 * #include "mbed.h"
 * #include "TTU_CSC1300.h"
 * 
 * TextLCD lcd(TextLCD::LCD_CURSOR_ON_BLINKING_ON);     // select your cursor style
 * 
 * int main() {
 *     lcd.printf("Hello World!\n");
 * }
 * @endcode
 */
class TextLCD : public Stream {
public:

    typedef enum {
        LCD_CURSOR_OFF_BLINKING_OFF = 0,
        LCD_CURSOR_OFF_BLINKING_ON = 0x01,        
        LCD_CURSOR_ON_BLINKING_OFF = 0x02,
        LCD_CURSOR_ON_BLINKING_ON = 0x03,
        LCD_CURSOR_DEFAULT = 0x03
    } LCDCursor;
  
    /** Constructor to create a TextLCD interface
     *
     * @param cur   Desired behavior of LCD cursor
     *
     * @see setCursor()
     * @see TextLCD::LCD_CURSOR_OFF_BLINKING_OFF    No cursor and no blinking block
     * @see TextLCD::LCD_CURSOR_OFF_BLINKING_ON     No cursor and but blinking block
     * @see TextLCD::LCD_CURSOR_ON_BLINKING_OFF     Underline cursor and no blinking block
     * @see TextLCD::LCD_CURSOR_ON_BLINKING_ON      Both underlinecursor and blinking block
     */ 
    TextLCD(LCDCursor cur);      // default constructor
 
#if DOXYGEN_ONLY
    /** Write a character to the LCD at the current cursor location
     *
     * @param c The character to write to the display
     *
     * @see setLocation()
     */
    int putc(int c);
 
    /** Write a formated string to the LCD at the curren cursor location
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     *
     * @note    A string is denoted whthin double-quotes.
     */
    int printf(const char* format, ...);
#endif
 
    /** Locate cursor to a screen row and column
     *
     * @param row     The vertical position from the top, indexed from 0
     * @param column  The horizontal position from the left, indexed from 0     
     */
    void setLocation(int column, int row);
 
    /** Clear the screen and locate to 0,0 */
    void cls();
 
    /** Right a single character to a specific location
     *
     * @param row     The vertical position from the top, indexed from 0
     * @param column  The horizontal position from the left, indexed from 0     
     * @param c       The character to display on-screen
     *
     * @note    A character is denoted within single-quotes (apostrophes)
     */
    void setCharacter(int row, int column, int c);

    /** Change the LCD cursor mode
     *
     * @param c     A LCDCursor type denoted the cursor and blink
     *
     * @see TextLCD::LCD_CURSOR_OFF_BLINKING_OFF
     * @see TextLCD::LCD_CURSOR_OFF_BLINKING_ON
     * @see TextLCD::LCD_CURSOR_ON_BLINKING_OFF
     * @see TextLCD::LCD_CURSOR_ON_BLINKING_ON
     */
    void setCursor(LCDCursor c);
 
protected:
 
    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();
 
    int address(int column, int row);
    void writeNibble(char incoming, unsigned char rs);
    void writeCommand(int command);
    void writeData(int data);
 
    int _column;
    int _row;
    int _maxColumns;
    int _maxRows;
    int _cursorState;
    
};
 
 
/* F031K6 + Edubase-V2 7-segment LED Library
 * for the 4 seven-segment LEDs inteface connected via two 74HC595s using SPI
 *
 * Copyright (c) 2020, jwbruce   (jwbruce@tntech.edu)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
/**  A 7-segment LED interface for driving 4 seven-segment LEDs connected
 *   via two cascaded 74HC595s using the STM32F031K6 SPI peripheral.
 *   Specifically, the Nucleo32 F031K6 boards plugged into the Edubase-V2
 *   via the adapter used at Tennessee Tech.
 *
 *   Supports only the four seven-segment displays (DISP1-DISP4) on
 *   the EduBase-V2 board
 *
 * @code
 * #include "mbed.h"
 * #include "TTU_CSC1300.h"
 * 
 * SSLED     seg7(4, ~0x00);    // blank DISP4 on Edubase-V2 board
 * 
 * int main() {
 *     while(1) {
 *         seg7.setDisplay( 1, ~0x3F);      // display "0"
 *         wait(1.0);
 *         seg7.setDisplay( 2, ~0x06);      // display "1"
 *         wait(1.0);
 *         seg7.setDisplay( 3, ~0x5B);      // display "2"
 *         wait(1.0);
 *         seg7.setDisplay( 4, ~0x4F);      // display "3"
 *         wait(1.0);
 *     } // end while()
 * } // end main()
 * @endcode
 */
class SSLED {
public:
    /** Constructor to create the 4-digit seven-segment display interface
     *
     * @param display   integer (1-4) selected desired 7-segment display DISP1-DISP4
     * @param pattern   byte containing bitmask of to display (zeroed bits illuminate)
     *
     * @see     setDisplay()
     */ 
    SSLED(int display, int pattern);      
 
    /** Write a pattern to a chosen display
     *
     * @param display   integer (1-4) selected desired 7-segment display DISP1-DISP4
     * @param pattern   byte containing bitmask of to display (zeroed bits illuminate)
     *
     * @note    MSb in pattern is decimal point segment, then gfedcba (with segment
     *          a as LSb). Zeroes in the pattern bitmask are illuminated.
     *          For example, pattern ~0x66 will illuminate the number "4".
     *
     * @see     https://en.wikipedia.org/wiki/Seven-segment_display
     */ 
    void setDisplay(int display, int pattern); // change patterns to blank and write the displays
};
 
 

#endif