#include "mbed.h"
#include "ADS8568_ADC.h"
#include "FastPWM.h"
#include "MODSERIAL.h"

#define MEAS_DELAY       60  // measurement delay after turning on FET (us)
#define LOG_INTERVAL   1000  // log file interval (ms)
#define START_DELAY    1000  // pause for startup (ms)
#define N_STEPS          20
#define BUFFER_SIZE    4096

//UID lookup address and pointer
#define UID_ADDR 0x1FFF7A10
unsigned long *uid = (unsigned long *) UID_ADDR; 

//UID and drive board calibration table
#define UID_TABLE_LENGTH 4

int drive_board_serial_number[UID_TABLE_LENGTH] =
  {1,
   2,
   3,
   4};

unsigned long drive_board_uid[UID_TABLE_LENGTH][3] =
 {{0x005B0060, 0x32375101, 0x32363531},
  {0x00000000, 0x00000000, 0x00000000},
  {0x00000000, 0x00000000, 0x00000000},
  {0x00000000, 0x00000000, 0x00000000}};

float drive_board_cal[UID_TABLE_LENGTH][2] = 
 {{0.0, 1.0},
  {0.0, 1.0},
  {0.0, 1.0},
  {0.0, 1.0}}; 
  

MODSERIAL pc(PA_9, PA_10, BUFFER_SIZE); //mcu TX, RX, BUFFER_SIZE byte TX and RX buffers
ADS8568_ADC adc(PB_15, PB_14, PB_13, PB_12, PC_15, PC_0, PC_1, PC_2, PC_3);
I2C i2c(PB_7, PB_8);            //SDA, SCL
Timer timer;
DigitalIn adc_busy(PA_8);                   //Busy interrupt sig#

//Heater Control
FastPWM drive_1(PC_9);
FastPWM drive_2(PC_8);
FastPWM guard_1(PC_7);
FastPWM guard_2(PC_6);

//Indicator LEDs
DigitalOut hb_led(PC_13);       //Green
DigitalOut led_0(PC_4);         //Red
DigitalOut led_1(PC_5);         //Green

//User buttons
DigitalIn user_0(PB_0);
DigitalIn user_1(PB_1);

BusOut converts(PC_0, PC_1, PC_2, PC_3);

int main() {
    int eTime;
    int v[2], curr[2];
    double r_adc[2], r_ohm[2];

    int i_port[2] = {0,2};
    int v_port[2] = {1,3};
     
    pc.baud(115200);
    adc.init();
    
    // Initialsation - all heaters off
    drive_1.prescaler(1);
    drive_1.period_ticks(1000);
    drive_1.pulsewidth_ticks(0);

    guard_1.prescaler(1);
    guard_1.period_ticks(1000);
    guard_1.pulsewidth_ticks(0);

    drive_2.prescaler(1);
    drive_2.period_ticks(1000);
    drive_2.pulsewidth_ticks(0);

    guard_2.prescaler(1);
    guard_2.period_ticks(1000);
    guard_2.pulsewidth_ticks(0);
   
    pc.printf("\r\nUnique ID: %08X %08X %08X \r\n", uid[0], uid[1], uid[2]);   
    int i_board = 0;  
    for (int i = 0; i < UID_TABLE_LENGTH; i++)
    {
      if (uid[0]==drive_board_uid[i][0] && uid[1]==drive_board_uid[i][1] && uid[2]==drive_board_uid[i][2])
        {
            i_board = drive_board_serial_number[i];
            i = UID_TABLE_LENGTH;
        }
    }
    
    if (i_board > 0) pc.printf("Drive board: BRD%d\n",i_board);
    else pc.printf("Drive board UID match not found\n");

    pc.printf("iStep,      eTime,        I1,         V1,     R1_adc,     R1_ohm,       I2,         V2,     R2_adc,     R2_ohm\n");
    
    wait_ms(START_DELAY);
    timer.start();

    for (int iStep=0; iStep<N_STEPS; iStep++) {

        eTime = timer.read_ms();
        pc.printf("%5d, %10d,", iStep, eTime);

        for (int iHeater=0; iHeater <2; iHeater++) {
            // measure heater
            if (iHeater==0)
                drive_1.pulsewidth_ticks(1000);
            else
                drive_2.pulsewidth_ticks(1000);
            wait_us(MEAS_DELAY);
        
            //Start ADC conversion
            adc.start_conversion(15);

            //Wait until ADC is free
            while(adc_busy == 1) {
                wait_us(1);
            }

            //Turn off heater
            if (iHeater==0)
                drive_1.pulsewidth_ticks(0);
            else
                drive_2.pulsewidth_ticks(0);
            wait_us(MEAS_DELAY);
        
            //Get ADC values
            adc.read_channels();
            curr[iHeater] = adc.read_channel_result(i_port[iHeater]);
            v[iHeater] = adc.read_channel_result(v_port[iHeater]);
            r_adc[iHeater] = (float)v[iHeater]/(float)curr[iHeater];
            if (i_board > 0)
                r_ohm[iHeater] = drive_board_cal[i_board][0] + r_adc[iHeater]*drive_board_cal[i_board][1];
            else
                r_ohm[iHeater] = 0.0;
                

            //Write output for iHeater
            pc.printf("%10d, %10d, %10.6f, %10.6f", curr[iHeater], v[iHeater], r_adc[iHeater], r_ohm[iHeater]);
        }
        pc.printf("\n");
        wait_ms(LOG_INTERVAL);
    }
 }
