#include "MPL115A1.h"

// These values correspond to the values described here:
// https://www.nxp.com/docs/en/data-sheet/MPL115A1.pdf

#define a0FracBits  3
#define b1FracBits  13
#define b2FracBits  14
#define c12FracBits 22
#define a0FracMask  0x0007
#define b1FracMask  0x1FFF
#define b2FracMask  0x3FFF
#define c12FracMask 0x1FFF

static float convertA0(uint16_t a0);
static float convertB1(uint16_t b1);
static float convertB2(uint16_t b2);
static float convertC12(uint16_t c12);

MPL115A1::MPL115A1(SPI &spi, NRF52_DigitalOut &cs) : spi(spi), cs(cs) {}

float MPL115A1::getA0()
{
    spi.write(0x88); // MSB a0
    uint16_t a0 = spi.write(0x00);   
    a0 = a0 << 8;
    wait_ms(1); 
    spi.write(0x8A); // LSB a0
    a0 |= spi.write(0x00);
    float result = convertA0(a0);
    wait_ms(1);
    return result;
}

float MPL115A1::getB1()
{
    spi.write(0x8C); // MSB b1
    uint16_t b1 = spi.write(0x00);   
    b1 = b1 << 8;
    wait_ms(1);
    spi.write(0x8E); // LSB b1
    b1 |= spi.write(0x00);
    float result = convertB1(b1);
    wait_ms(1);
    return result;
}

float MPL115A1::getB2()
{
    spi.write(0x90); // MSB b2
    uint16_t b2 = spi.write(0x00);   
    b2 = b2 << 8;
    wait_ms(1);
    spi.write(0x92); // LSB b2
    b2 |= spi.write(0x00);
    float result = convertB2(b2);
    wait_ms(1);
    return result;
}

float MPL115A1::getC12()
{
    spi.write(0x94);        // MSB c12
    uint16_t c12 = spi.write(0x00);  
    c12 = c12 << 8;
    wait_ms(1);
    spi.write(0x96);        // LSB c12
    c12 |= spi.write(0x00);
    float result = convertC12(c12);
    wait_ms(1);
    return result;
}

uint16_t MPL115A1::getPadc()
{
    uint16_t padc;
    spi.write(0x80);
    padc = spi.write(0x00);  // MSB Padc
    padc = padc << 8;
    spi.write(0x82);
    padc |= spi.write(0x00); // LSB Padc
    padc = padc >> 6;
    return padc;
}

uint16_t MPL115A1::getTadc()
{
    uint16_t tadc;
    spi.write(0x84);                
    tadc = spi.write(0x00);  // MSB Padc
    tadc = tadc << 8;
    spi.write(0x86);
    tadc |= spi.write(0x00); // LSB Padc
    tadc = tadc >> 6;
    return tadc;
}

float MPL115A1::getPressure(){
    cs = 0;
    float a0  = getA0();
    float b1  = getB1();
    float b2  = getB2();
    float c12 = getC12();
    spi.write(0x00);
    cs = 1;
    
    cs = 0;
    spi.write(0x24);        // Start conversion
    spi.write(0x00);
    cs = 1;
    wait_ms(3);
    
    cs = 0;
    uint16_t padc = getPadc();
    uint16_t tadc = getTadc();
    spi.write(0x00);
    cs = 1;
    
    float pcomp = a0 + (b1 + c12 * tadc) * padc + b2 * tadc;
    float pressure = pcomp * ((115-50)/(1023.0)) + 52;
    pressure *= 10; // Calculate in hPa
    return pressure;
}

static float convertA0(uint16_t a0){
    float tempA0;
    float tempFrac;
    int tempA0Int;
    uint8_t negativeFlag = 0;
    
    if(a0 & 0x8000){
        a0 ^= 0xFFFF; // Transform from 2's complement
        a0 -= 1;
        negativeFlag = 1;
    }
    
    tempA0Int = a0 & 0x7FF8;
    tempA0Int = tempA0Int >> a0FracBits;
    
    tempA0 = tempA0Int * 1.0;    // Int part
    tempFrac = (1.0/(1<<3));
    tempFrac *= (a0 & a0FracMask);
    tempA0 = tempA0 + tempFrac;
    
    if(negativeFlag) tempA0 = -tempA0;
    
    return tempA0;
}

static float convertB1(uint16_t b1){
    float tempB1;
    float tempFrac;
    int tempB1Int;
    uint8_t negativeFlag = 0;
    
    if(b1 & 0x8000){
        b1 ^= 0xFFFF;
        b1 -= 1;
        negativeFlag = 1;
    }
    
    tempB1Int = b1 & 0x6000;
    tempB1Int = tempB1Int >> b1FracBits;
    
    tempB1 = tempB1Int * 1.0;
    tempFrac = (b1 & b1FracMask) * (1.0/(1<<b1FracBits));
    tempB1 = tempB1 + tempFrac;
    
    if(negativeFlag) tempB1 = -tempB1;
    
    return tempB1;
}

static float convertB2(uint16_t b2){
    float tempB2;
    float tempFrac;
    int tempB2Int;
    uint8_t negativeFlag = 0;
    
    if (b2 & 0x8000){
        b2 ^= 0xFFFF;
        b2 -= 1;
        negativeFlag = 1;
    }
    
    tempB2Int = b2 & 0x4000;
    tempB2Int = tempB2Int >> b2FracBits;
    
    tempB2 = tempB2Int * 1.0;
    tempFrac = (b2 & b2FracMask) * (1.0/(1<<b2FracBits));
    tempB2 = tempB2 + tempFrac;
    
    if(negativeFlag) tempB2 = -tempB2;
    
    return tempB2;
}

static float convertC12(uint16_t c12){
    float tempC12;
    float tempFrac;
    uint8_t negativeFlag = 0;
    
    c12 = c12 >> 2;
    
    if (c12 & 0x2000){
        c12 ^= 0xFFFF;
        c12 -= 1;
        negativeFlag = 1;
    }
    
    tempC12 = 0.000000000;
    tempFrac = (c12 & c12FracMask) * (1.0/(1<<c12FracBits));
    tempC12 = tempC12 + tempFrac;
    
    if(negativeFlag) tempC12 = -tempC12;
    
    return tempC12;
}
