/**
 * SAP Project by aconno
 * Made by Jurica @ aconno
 * More info @ aconno.de
 *
 */

#include "mbed.h"
#include "ATCmdParser.h"
#include "FileHandle.h"
#include "gnss.h"
#include "aconnoConfig.h"
#include "uBloxSara.h"
#include "aconnoHelpers.h"
#include "Lis2dh12.h"
#include "tasks/tasks.h"

#define RED_LED_PIN			(PE_3)
#define GREEN_LED_PIN		(PE_4)
#define BLUE_LED_PIN		(PE_1)
#define ALARM_OFF           (0)
#define ALARM_ON            (1)

volatile MainStates state = STATE_IDLE;

/* File handler */
FileHandle *fh;

/* AT Command  Parser handle */
ATCmdParser *at;
GnssSerial gnss;	// Serial interface (not I2C)
char locationGlobal[UDP_MSG_SIZE_B];
bool gotGnssFlag = false;
DigitalIn userButton(USER_BUTTON);
DigitalOut gnssPower(PE_0, 0);

DigitalOut redLed(RED_LED_PIN);
DigitalOut blueLed(BLUE_LED_PIN);
DigitalOut greenLed(GREEN_LED_PIN);
DigitalOut alarm(PE_14, 0);
InterruptIn buttonInt(USER_BUTTON);

static bool wasAccInt = false;
static bool wasButtPressed = false;
static bool longPress = false;
Timer buttonTimer;
Thread timerThread;
Thread idleState; 
Thread alarmState; 
Thread alarmOffState; 
Thread movementState; 
Thread gnssLocation;

void my_bsp_init()
{
    redLed = 1;
    greenLed = 1;
    blueLed = 1;
    alarm = ALARM_OFF;
}

void buttonRiseHandler(void)
{
    buttonTimer.reset();
    buttonTimer.start();
    timerThread.signal_set(BUTTON_PRESSED_SIGNAL);
}

void timerCallback(void)
{
    float secondsPassed;
    while(true)
    {
        bool nextState = false;
        secondsPassed = 0;
        Thread::signal_wait(BUTTON_PRESSED_SIGNAL);
        Thread::signal_clr(BUTTON_PRESSED_SIGNAL);
        while(!nextState)
        {
            secondsPassed = buttonTimer.read();
            if(state != STATE_ALARM && secondsPassed > START_ALARM_S)
            {
                wasButtPressed = true;
                buttonTimer.stop();
                buttonTimer.reset();
                nextState = true;
            }
            else if(state == STATE_ALARM && secondsPassed > STOP_ALARM_S)
            {
                longPress = true;
                buttonTimer.stop();
                buttonTimer.reset();
                nextState = true;
            }
        }
    }
}

int main()
{
    bool success = false;
    MainStates nextState = STATE_IDLE;
    
    buttonInt.rise(buttonRiseHandler);
    my_bsp_init();

    printf("Initialising UART for modem communication: ");
    fh = new UARTSerial(MDMTXD, MDMRXD, 9600);
    printf("...done\r\n");

    printf("Initialising the modem AT command parser: ");
    at = new ATCmdParser(fh, OUTPUT_ENTER_KEY, AT_PARSER_BUFFER_SIZE,
                         AT_PARSER_TIMEOUT, false);
    printf("...done\r\n");

	gnssPower = 1;
    wait_ms(500);

    if(gnss.init())
	{
        printf("Cold start... waiting to get first location data\r\n");     
        getGPSData(locationGlobal, &gnss);
        printf("I have the location.\r\n");
        gotGnssFlag = true;
        // Turn green led only
        greenLed = 0;
        redLed = 1;
	}
	else
	{
        printf("Unable to initialise GNSS.\r\n");
    }

    Udp udp("52.215.10.12", 3334);
    UBloxSara sara(at, udp);
    printf("Initializing the modem: \r\n");
	sara.setup();
    sara.connectNB();
    printf("done...\r\n");

    alarm = ALARM_OFF;
    greenLed = 1; 
    blueLed = 0;
    myParams_t myParams;
    myParams.gnss = &gnss;
    myParams.sara = &sara;
    
    timerThread.start(timerCallback);
    idleState.start(idleCallback);
    alarmState.start(callback(alarmCallback, &myParams));
    alarmOffState.start(alarmOffCallback);
    movementState.start(movementCallback);
    gnssLocation.start(callback(gnssLocationCallback, &gnss));
    
    char commandbuffer[100];
	
    while(1)
	{
        switch(state)
        {
            case STATE_IDLE:
				if(wasAccInt)
                {
                    wasAccInt = false;
                    nextState = STATE_LIS_DETECTION;
                }
                if(wasButtPressed)
                {
                    wasButtPressed = false;
                    nextState = STATE_ALARM;
                    alarmState.signal_set(ALARM_SIGNAL);
                    greenLed = 1;
                    redLed = 0;
                    blueLed = 1;
                }
            	break;
            case STATE_ALARM:
				if(longPress)
                {
                    redLed = 1;
                    longPress = false;
                    nextState = STATE_ALARM_OFF;
                    alarmOffState.signal_set(ALARM_OFF_SIGNAL);
                }
                else
                {
                    alarm = ALARM_ON;
                }
            	break;
            case STATE_ALARM_OFF:
                nextState = STATE_IDLE;
                idleState.signal_set(IDLE_SIGNAL);
                alarm = ALARM_OFF;
                blueLed = 0;
				break;
            case STATE_LIS_DETECTION:
            	break;
        }
        state = nextState;
        wait_ms(125);
	}
}
