/* 
 * aconno.de
 * Made by Jurica Resetar
 * All right reserved
 *
 */

#include "stdio.h"
#include "mbed.h"
#include "ble/BLE.h"
#include "acd52832_bsp.h"
#include "GapAdvertisingData.h"
#include "nrf52_uart.h"

#define SLEEP_TIME      (2.0)           /* Sleep time in seconds */
#define WAKE_UP_TIME    (2000)          /* Awake time in ms */
#define MSD_SIZE        (1)             /* Manufacturer Specific Data lenght (in B) */
#define ADV_INTERVAL    (1000)          /* Advertising interval in ms */
#define TX_POWER        (4)             /* TX power (in dB) */
#define GO_TO_SLEEP     (0)             /* Sleep flag: 0 -> Device will not go to sleep, 1 -> Will go to sleep mode */

#define TX              (p27)
#define RX              (p31)

bool SLEEP = true;

BLE &ble = BLE::Instance();
GapAdvertisingData adv_data = GapAdvertisingData();
uint8_t MSD[MSD_SIZE] = {0x00};

/**
 * Restart Advertising on disconnection
 */
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params){
    BLE::Instance().gap().startAdvertising();
}

/**
 *  Function for waking the core up
 */
void wakeMeUp(void){
    SLEEP = false;
}

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error){
    /* Avoid compiler warnings */
    (void) ble;
    (void) error;
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params){
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, (uint8_t *)MSD, MSD_SIZE);
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);    
    ble.gap().setAdvertisingInterval(ADV_INTERVAL);
    ble.gap().startAdvertising();    
}

void updateData(){    
    static uint8_t simple_counter = 0;
    
    MSD[0] = simple_counter++;
    adv_data = ble.getAdvertisingData();
    adv_data.updateData(adv_data.MANUFACTURER_SPECIFIC_DATA, MSD, MSD_SIZE);
    ble.setAdvertisingData(adv_data);
}

char buffer[255] = {};
NRF52_UART serial = NRF52_UART(TX, RX, Baud9600);
#include "acd_nrf52_saadc.h"

int main(void){
    uint8_t buffer_s;
    int printBufferLen;

    Ticker ticker;
    ticker.attach(wakeMeUp, SLEEP_TIME);   // Wake the device up   
    
    ble.init(bleInitComplete);
    ble.gap().setTxPower(TX_POWER);        // Set TX power to TX_POWER
    
    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while(ble.hasInitialized()  == false) { /* spin loop */ }   
    
    NRF52_SAADC analogIn;
    analogIn.addChannel(9); // vdd
    analogIn.addChannel(5); // potentiometer
    analogIn.calibrate(); // nije previse bitno
    while(1){
        buffer_s = 0x0C;
        serial.send(&buffer_s, sizeof(buffer_s));
        analogIn.updateData();
        
        printBufferLen = sprintf(buffer, "VDD           voltage: %f\r\n", (analogIn.getData()[0])*(1.0/1024)*3.6);
        serial.send(buffer, printBufferLen);
        
        printBufferLen = sprintf(buffer, "POTENTIOMETER voltage: %f\r\n", (analogIn.getData()[1])*(1.0/1024)*3.6);
        serial.send(buffer, printBufferLen);
        wait_ms(3000);
    }
    
    while(true){
        if (SLEEP && GO_TO_SLEEP){
            ble.gap().stopAdvertising();
            sleep();
            ble.waitForEvent();
        }
        else{
            // I'm awake
            updateData();
            ble.gap().startAdvertising();
            wait_ms(WAKE_UP_TIME);
            SLEEP = true;
        }
    }
}
