/* 
 * Copyright (c) 2017 Aconno. All Rights Reserved.
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRANTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 */
 
#include "mbed.h"
#include "acd52832_bsp.h"
#include "GDEP015OC1.h"
#include "pictures.h"
#include "acd_nrf52_saadc.h"

#define DEBUG                   (1)
#define ADC_MAX_VALUE           (8192)
#define ADC_REF_VOLTAGE         (3.3f)
#define VOLTAGE_DIVIDER_RATION  (7.0606)
#define VOLTAGE_LOSS            (0.4)
#define CURRENT_FACTOR          (36.0)
#define BATTERY_CHARGED_V       (13)
#define BATTERY_MAX_V           (12.6)
#define BATTERY_MIN_V           (11.85)
#define BATTERY_STEP            (0.15) 
#define QUICK_CURRENT           (float)(0.8)
#define LOW_CURRENT             (float)(0.4)
#define NO_CURRENT              (float)(0.01)
#define NUM_OF_MEASUREMENTS     (1)
#define NO_CURRENT_STATE        (0)
#define LOW_CURRENT_STATE       (1)
#define QUICK_CURRENT_STATE     (2)
    
#define _CH_4                   (9)
#define _CH_3                   (8)
#define _CH_2                   (7)
#define _CH_1                   (6)   
#define _BS_5                   (5)
#define _BS_4                   (4)
#define _BS_3                   (3)
#define _BS_2                   (2)
#define _BS_1                   (1)
#define _BS_E                   (0)

#define ANALOG_PIN_P     (4)
#define ANALOG_PIN_N     (5)
#define NUM_OF_DIFF_ADCs (1)

SPI spi(PIN_EPD_MOSI, NC, PIN_EPD_SCK, NC);
GDEP015OC1 epd = GDEP015OC1(spi, PIN_EPD_CS, PIN_EPD_DC, PIN_EPD_RST, PIN_EPD_BUSY);    

uint8_t get_battery_level(float battery_voltage){
        if (battery_voltage > BATTERY_CHARGED_V * 4){
                return _CH_4;
        }
        else if ((battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP * 3) && (battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP  * 4)){
                return _CH_3;
        }
        else if ((battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP * 2) && (battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP  * 3)){
                return _CH_2;
        }
        else if ((battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP * 1) && (battery_voltage > BATTERY_CHARGED_V + BATTERY_STEP * 2)){
                return _CH_1;
        }
        else if((battery_voltage > BATTERY_MAX_V - BATTERY_STEP) && (battery_voltage < BATTERY_CHARGED_V)){
                return _BS_5;
        }
        else if((battery_voltage > BATTERY_MAX_V - BATTERY_STEP * 2) && (battery_voltage < BATTERY_MAX_V - BATTERY_STEP * 1)){
                return _BS_4;
        }
        else if((battery_voltage > BATTERY_MAX_V - BATTERY_STEP * 3) && (battery_voltage < BATTERY_MAX_V - BATTERY_STEP * 2)){
                return _BS_3;
        }
        else if((battery_voltage > BATTERY_MAX_V - BATTERY_STEP * 4) && (battery_voltage < BATTERY_MAX_V - BATTERY_STEP * 3)){
                return _BS_2;
        }
        else if((battery_voltage > BATTERY_MAX_V - BATTERY_STEP * 5) && (battery_voltage < BATTERY_MAX_V - BATTERY_STEP * 4)){
                return _BS_1;
        }
        else if(battery_voltage < BATTERY_MAX_V - BATTERY_STEP * 5){
                return _BS_E;
        }
    return 0;
}

uint8_t get_current_level(float current_value, uint8_t usb_state){
    if(current_value < NO_CURRENT)
        return NO_CURRENT_STATE;
    else if(current_value > NO_CURRENT && current_value < LOW_CURRENT)
        return LOW_CURRENT_STATE;
    else if(current_value > LOW_CURRENT && current_value < QUICK_CURRENT){
        // Determine current charger state depends on previous state
        if(usb_state == LOW_CURRENT_STATE)
            return LOW_CURRENT_STATE;
        else if(usb_state == QUICK_CURRENT_STATE)
            return QUICK_CURRENT_STATE;
    }
    else if(current_value > QUICK_CURRENT)
        return QUICK_CURRENT_STATE;
    return NO_CURRENT_STATE;
}

int main(){
    NRF52_SAADC *diffADCs[NUM_OF_DIFF_ADCs];
    /* Declare your ADCs here */
    /* Change NUM_OF_DIFF_ADCs in the header */
    diffADCs[0] = new NRF52_SAADC(ANALOG_PIN_P, ANALOG_PIN_N);
    
    //NRF52_SAADC usb1(7); // P29, AIN5
    //NRF52_SAADC usb2(6); // p30, AIN6

    char low_string[25] = "LOW";
    char quick_string[25] = "QUICK";
    float adc1_mean=0, adc2_mean=0, adc3_mean=0;
    float battery_voltage = 0;
    float usb1_current = 0, usb2_current = 0;
    int count = 0;
    uint8_t battery_level = 0;
    uint8_t old_battery_level;
    uint8_t usb1_current_state = 0, usb2_current_state = 0;
    uint8_t usb1_old_state = 0, usb2_old_state = 0;
    uint8_t change_flag = 1;
    
    #if DEBUG
        char buffer[25] = {0};
        float old_battery_voltage = 0;
        float old_usb1_current = 0;
        float old_usb2_current = 0;
    #endif
        
    epd.empty();
    epd.writeFull(); 
        
    while(true){   
        
        adc1_mean += diffADCs[0]->read();
        adc2_mean += 1; //usb1.read();
        adc3_mean += 1; //usb2.read();
        count ++;
                            
        if (count == NUM_OF_MEASUREMENTS){ 
     
            adc1_mean /= NUM_OF_MEASUREMENTS;
            adc2_mean /= NUM_OF_MEASUREMENTS;
            adc3_mean /= NUM_OF_MEASUREMENTS;
            count = 0;
                
            //voltage[i] = ((float)(diffADCs[i]->read()))*(3.3f/8192.0);    // Convert raw data into voltage
            battery_voltage = ((float)(diffADCs[0]->read()*(3.3f/8192.0)));
            //battery_voltage = ((float)(diffADCs[0]->read()*(3.3f/8192.0)));
            //battery_voltage = adc1_mean*(ADC_REF_VOLTAGE/ADC_MAX_VALUE);
            usb1_current = (adc2_mean*(ADC_REF_VOLTAGE/(ADC_MAX_VALUE*2))) * CURRENT_FACTOR;
            usb2_current = (adc3_mean*(ADC_REF_VOLTAGE/(ADC_MAX_VALUE*2))) * CURRENT_FACTOR;
                    
            battery_level = get_battery_level(battery_voltage);
            if(battery_level != old_battery_level || change_flag){
                
                old_battery_level = battery_level;
                change_flag = 1;
                switch(battery_level){
                    case(0):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_E[x], x);
                            break;
                            }
                    case(1):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_1[x], x);
                            break;
                            }
                    case(2):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_2[x], x);
                            break;
                            }
                    case(3):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_3[x], x);
                            break;
                            }
                    case(4):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_4[x], x);
                            break;
                            }
                    case(5):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(BS_5[x], x);
                            break;
                            }
                    case(6):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(charged_1[x], x);
                            break;
                            }
                    case(7):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(charged_2[x], x);
                            break;
                            }
                    case(8):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(charged_3[x], x);
                            break;
                            }
                    case(9):{
                            for(uint16_t x=0;x<5000;x++)
                                epd.fill(charged_4[x], x);
                            break;
                            }
                    default: break;
                }
                epd.write();
            }
            
            usb1_current_state = get_current_level(usb1_current, usb1_old_state);
            usb2_current_state = get_current_level(usb2_current, usb2_old_state);
            
            if((usb1_old_state != usb1_current_state) || change_flag){
                usb1_old_state = usb1_current_state;
                switch(usb1_current_state){
                    case(NO_CURRENT_STATE):{
                        epd.writeString(quick_string, 25, 180, 1);  // Delete the old state
                        epd.writeString(low_string, 25, 180, 1);
                        break;
                    }
                    case(LOW_CURRENT_STATE):{
                        epd.writeString(quick_string, 25, 180, 1);  // Delete the old state
                        epd.writeString(low_string, 25, 180, 0);
                        break;
                    }
                    case(QUICK_CURRENT_STATE):{
                        epd.writeString(low_string, 25, 180, 1);    // Delete the old state
                        epd.writeString(quick_string, 25, 180, 0);
                        break;
                    }
                    default: break;
                }
                epd.write();                                            // Write string to the EPD
            }
            
            if((usb2_old_state != usb2_current_state) || change_flag){
                usb2_old_state = usb2_current_state;
                switch(usb2_current_state){
                    case(NO_CURRENT_STATE):{
                        epd.writeString(low_string, 135, 180, 1);       // Delete the old state
                        epd.writeString(quick_string, 135, 180, 1);     // Delete the old state
                        break;
                    }
                    case(LOW_CURRENT_STATE):{
                        epd.writeString(quick_string, 135, 180, 1);     // Delete the old state
                        epd.writeString(low_string, 135, 180, 0);
                        break;
                    }
                    case(QUICK_CURRENT_STATE):{
                        epd.writeString(low_string, 135, 180, 1);       // Delete the old state
                        epd.writeString(quick_string, 135, 180, 0);
                        break;
                    }
                    default: break;
                }
                epd.write();                                            // Write string to the EPD
                change_flag = 0;
            }

            
            #if DEBUG    
                // Print voltage and current values in debug mode
                sprintf(buffer, "Battery: %5.5fV", old_battery_voltage);     // Create a string
                epd.writeString(buffer,25,95,1);                        // Write new data to the buffer    
                epd.write();                                            // Write string to the EPD
                old_battery_voltage = battery_voltage;
                
                sprintf(buffer, "Battery: %5.5fV", battery_voltage);     // Create a string
                epd.writeString(buffer,25,95,0);                        // Write new data to the buffer    

                sprintf(buffer, "USB1: %5.5f", old_usb1_current);           // Create a string
                epd.writeString(buffer,5,190,1);                        // Write new data to the buffer    
                old_usb1_current = usb1_current;
                sprintf(buffer, "USB1: %5.5f", usb1_current);           // Create a string
                epd.writeString(buffer,5,190,0);                        // Write new data to the buffer    
    
                sprintf(buffer, "USB2: %5.5fA", old_usb2_current);           // Create a string
                epd.writeString(buffer,105,190,1);                        // Write new data to the buffer    
                old_usb2_current = usb2_current;
                sprintf(buffer, "USB2: %5.5fA", usb2_current);           // Create a string
                epd.writeString(buffer,105,190,0);                       // Write new data to the buffer    
                
                epd.write(); 
            #endif
            
            adc1_mean = 0;
            adc2_mean = 0;
            adc3_mean = 0;
        }
    }
}