/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 *
 * The library is modified to suite Si7006A20 Humidity and Temperature Sesor
 * Made by Jurica Resetar @ aconno
 * More info and contact: aconno.de
 *
 *******************************************************************************
 */

#ifndef _Si70006A20_H_
#define _Si70006A20_H_

#include "mbed.h"
#include "aconno_i2c.h"

/***** Definitions *****/
#define I2C_ADDR    (0x80)  // 1000000x
#define POLYVAL     (0x131)


class Si7006{
    public:
    
        /**
         * @brief   Measurement resolution.
         * @details Controls the resolution of the humidity and temperarure readings.
         */
        typedef enum {
            RH_12b_TEMP_14b = 0x0, ///< 12 bits for RH, 14 bits for Temp
            RH_8b_TEMP_12b = 0x1,  ///< 8 bits for RH, 12 bits for Temp
            RH_10b_TEMP_13b = 0x2, ///< 10 bits for RH, 13 bits for Temp
            RH_11b_TEMP_11b = 0x3, ///< 11 bits for RH, 11 bits for Temp
        } resolution_t;
            
        /**
         * Si7006 constructor
         *
         */
         Si7006(I2C *ic2);
    
        /**
         * @brief   Get Electronic ID.
         * @details Gets the Electronic ID of the connected device. Verifies the 
         *          ID with an 8-bit CRC.
         *
         * @param   Character buffer to store the id. Needs to be at least 8 bytes.
         * @returns 0 if no errors, -1 if error.
         */
        uint8_t getElectronicId(char *id);
    
        /**
         * @brief   Configure sample resolution.
         * @details Sets the number of bits used for humidity and temperature readings.
         * @param   resolution Enum for the resolution setting.
         * @returns 0 if no errors, -1 if error.
         */
        int configResolution(Si7006::resolution_t resolution);
    
        /**
         * @brief   Get temperature reading.
         * @details Initiates a temperature reading and blocks until reading has
         *          been calculated. 
         * 
         * @note    Will hold the I2C bus until reading is complete. Refer to datasheet
         *          for timing specifications.
         * 
         * @param   tempC Pointer for temperature reading. Unit is degree Celcius.
         * @returns 0 if no errors, -1 if error.
         */
        float getTemperature();
    
        /**
         * @brief   Check temperature reading.
         * @details Checks to see if the temperature reading has been completed.
                    Returns temperature if reading complete.
         * @note    Must call startTemperature() prior to calling this function. 
         * @param   tempC Pointer for temperature reading. Unit is degree Celcius.
         * @returns 0 if reading taken, -1 if reading pending.
         */
        float calculateTemperature(char *rawTemp);
    
        /**
         * @brief   Get humidity reading.
         * @details Initiates a humidity reading and blocks until reading has
         *          been calculated.
         *
         * @note    Will hold the I2C bus until reading is complete. Refer to datasheet
         *          for timing specifications.
         * 
         * @param   humidx10 Pointer for humidity reading. Unit is 1/10th percent.
         * @returns 0 if no errors, -1 if error.
         */
        float getHumidity();
    
        /**
         * @brief   Get humidity reading.
         * @details Initiates a humidity reading and blocks until reading has
         *          been calculated.
         *
         * @note    Will hold the I2C bus until reading is complete. Refer to datasheet
         *          for timing specifications.
         * 
         * @param   humid Pointer for humidity reading. Unit is percent.
         * @returns 0 if no errors, -1 if error.
         */
        float checkHumidity(char *rawHumidity);

        /**
         * @brief   Get temperature from humidity reading.
         * @details Gets temperature reading from previous humidity reading.
         * @note    Must call startHumidity() prior to calling this function. 
         * @param   tempC Pointer for temperature reading. Unit is degree Celcius.
         * @returns 0 if reading taken, -1 if reading pending.
         */
        int getPrevTemperature(float* tempC);
    
        /**
         * @brief   Get temperature from humidity reading.
         * @details Gets temperature reading from previous humidity reading.
         * @note    Must call startHumidity() prior to calling this function. 
         * @param   tempCx10 Pointer for temperature reading. Unit is 1/10th degree Celcius.
         * @returns 0 if reading taken, -1 if reading pending.
         */
        int getPrevTemperature(int16_t *tempCx10);
    
        /**
         * @brief   Get firmware revision.
         * @details Reads the firmware revision, refer to datasheet for codes.
         * @param   rev Pointer to store firmware revision.
         * @returns 0 if no errors, -1 if error.
         */
        int getRev(char *rev);
    
        /**
         * @brief   Control heater.
         * @details Enable or disable the heater.
         * @param   enable True to enable heater, false to disable.
         * @returns 0 if no errors, -1 if error.
         */
        int heater(bool enable);
        
    private:
    
        char crc8(char value, char seed);
        aconno_i2c i2c;
        bool i2c_owner;
};

#endif /* _Si7006_H_ */
