/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "LEDService.h"

#define NO_COMMAND          (0)
#define ZOOM_IN             (1)
#define ZOOM_OUT            (2)
#define FOCUS_NEAR          (3)
#define FOCUS_FAR           (4)
#define RECORD              (5)

#define LANC_COMMAND_PIN    (p2)   // Pin connected to Tr to pull lanc bus down/up     2
#define LANC_PIN            (p3)   // Lanc bus pin (to scan for START/STOP bits)        3


#define LANC_BIT_DURATION   (105)   // One LANC bit duration in us
#define LANC_MIN_DELAY      (5)
#define NUMBER_OF_REPETITIONS (4)

#define REC_CMD_1           (0x18)
#define REC_CMD_2           (0x33)
#define ZOOM_IN_CMD_4_1     (0x28)
#define ZOOM_IN_CMD_4_2     (0x08)
#define ZOOM_OUT_CMD_4_1    (0x28)
#define ZOOM_OUT_CMD_4_2    (0x18)
#define FOCUS_NEAR_1        (0x28)
#define FOCUS_NEAR_2        (0x47)
#define FOCUS_FAR_1         (0x28)
#define FOCUS_FAR_2         (0x45)

DigitalOut alivenessLED(p26, 0);
DigitalOut led_red(p22, 0);
DigitalOut led_blue(p23, 0);
DigitalOut led_green(p24, 0);

Serial uart(LANC_COMMAND_PIN, NC, 9600); // tx, rx-> NOT used

void send_command(uint8_t command_1, uint8_t command_2);

DigitalIn  lanc_pin        LANC_PIN;

Timer timer;

const static char     DEVICE_NAME[] = "Lanc";
static const uint16_t uuid16_list[] = {LEDService::LED_SERVICE_UUID};

static EventQueue eventQueue(
    /* event count */ 10 * /* event size */ 32
);

LEDService *ledServicePtr;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params){
    (void) params;
    BLE::Instance().gap().startAdvertising();
}

void blinkCallback(void){
    alivenessLED = !alivenessLED; /* Do blinky on LED1 to indicate system aliveness. */
}

/**
 * This callback allows the LEDService to receive updates to the ledState Characteristic.
 *
 * @param[in] params
 *     Information about the characterisitc being updated.
 */
void onDataWrittenCallback(const GattWriteCallbackParams *params) {
    if ((params->handle == ledServicePtr->getValueHandle()) && (params->len == 1)) {
        //actuatedLED = *(params->data);
        led_red = 1;
        led_blue = 1;
        led_green = 1;
        if (*(params->data)==ZOOM_IN){
                led_red = 0;
                send_command(ZOOM_IN_CMD_4_1,ZOOM_IN_CMD_4_2);
            }
        else if (*(params->data)==ZOOM_OUT){
                led_blue = 0;
                send_command(ZOOM_OUT_CMD_4_1,ZOOM_OUT_CMD_4_2);
            }
        else if (*(params->data)==FOCUS_NEAR){
                led_green = 0;
                send_command(FOCUS_NEAR_1, FOCUS_NEAR_2);
            }
        else if (*(params->data)==FOCUS_FAR){
                led_green = 0;
                led_red = 0;
                send_command(FOCUS_FAR_1,FOCUS_FAR_2);
            }
        else if (*(params->data)==RECORD){
                led_blue = 0;
                led_red = 0;
                send_command(REC_CMD_1, REC_CMD_2);
        }
        else if (*(params->data)==NO_COMMAND){
                // No command
            }
    }
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error){
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params){
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);
    ble.gattServer().onDataWritten(onDataWrittenCallback);

    bool initialValueForLEDCharacteristic = false;
    ledServicePtr = new LEDService(ble, initialValueForLEDCharacteristic);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms. */
    ble.gap().startAdvertising();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

void init_lanc_camera(void){
    // Function will set bus to HIGH and hold it for 5ms
    
}

void init_lanc_bus(){
    // Function initialize lanc bus by pulling it up
    // Bus goes HIGH by setting command pin to LOW
    wait_ms(LANC_MIN_DELAY);
}

void send_command(uint8_t command_1, uint8_t command_2){
    uint8_t command_counter = 0;
    
    while(command_counter < NUMBER_OF_REPETITIONS){
        while(1){
            while (lanc_pin == 0);      // Waiting for HIGH
            timer.start();
            while(lanc_pin == 1);        // Waiting for LOW 
            timer.stop();
            if(timer.read_us()>LANC_MIN_DELAY){
                break;
            }
        }
    
    // Program is out of the while loop -> bus is LOW -> camera generated START BIT
    // START BIT should last for 104us
    //wait_us(104);  // -> NEMOJ čekat jer bi to značilo da će start bit trajati dvostruko duže
    // Now write data on LANC_COMMAND pin
    uart.putc(command_1 ^ 0xFF);
    
    // Put LANC bus back to HIGH
    // Make sure to be in the stop bit before the Byte 1
    wait_us(5); 
    
    // Look while lanc bus is HIGH for STOP bit
    while(lanc_pin == 1);
    
    uart.putc(command_2 ^ 0xFF);
    //wait_us(LANC_BIT_DURATION);
    //Write Byte 1 to the bus
    /*
    for(int i=0; i<8; i++){
        command_pin = (command_2 & (1<<i));
        wait_us(LANC_BIT_DURATION);
    }
    */
    // Put LANC bus back to HIGH
    command_counter++;
    
    /*
     *  Control Bytes 0 and 1 are written. Program does not care what happend in
     *  Bytes 2-7 and just waits for next start bit after a long pause to send 
     *  command again
     */  
    }
}


int main(){
    // Clear RGB led
    led_red = 1;
    led_green = 1;
    led_blue = 1;
    
    // init_lanc_bus();
    
    eventQueue.call_every(50, blinkCallback);
    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    while(1){
        send_command(REC_CMD_1,REC_CMD_2);
        wait_ms(1000);
    }
    
    eventQueue.dispatch_forever();

    return 0;
}
