/*******************************************************************************
* @file MAX5487.h

*******************************************************************************
*/

#ifndef MAX5487_H
#define MAX5487_H

#include "mbed.h"


/**
* @brief Dual Channel SPI Digital Potentiometers with Non Volatile Memory
*        for the MAX5487, MAX5488, MAX5489.
*
* @details The MAX5487 contains two SPI programmable potentiometers. 
* This driver enables setting the potentiometer wiper values as well
* as the nonvolatile memory. This driver is compatible with the
* MAX5487, MAX5488, MAX5489.
*
* @code 
* #include "mbed.h"
* #include "max32630fthr.h"
* #include "USBSerial.h"
* #include "MAX5487.h"
* MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);
* Serial daplink(P2_1, P2_0);
* USBSerial microUSB; 
* DigitalOut led1(LED1); // led to blink
* DigitalOut selectPin(P3_0); // Pin 6_0 is used to drive chip enable low
* SPI spi(P5_1, P5_2, P5_0); // mosi, miso, sclk

* int main()
* {
*    MAX5487 test(spi, selectPin);
*    spi.format(8,0);
*    spi.frequency(1000000);
*    test.writeCommand(MAX5487::Wiper_RegA, 0x55);
*
*      //... rest of application
* }
* @endcode
*/



class MAX5487{
    public:
    /**
    * @brief Commands supported by the potentiometer
    * @details write commands that don't involve copying 
    * require an 8 bit value specified. Copy commands don't
    * requre sending an 8 bit value.
    */ 
    typedef enum {
        Wiper_RegA = 0x01,                        //Position of Wiper A
        Wiper_RegB = 0x02,                        //Position of Wiper B
        NV_RegA = 0x11,                           //Non Volatile Register A 
        NV_RegB = 0x12,                           //Non Volatile Register B 
        Copy_Wiper_RegA_to_NV_RegA   = 0x21,      //Copy Wiper A position to Non Volatile Register A
        Copy_Wiper_RegB_to_NV_RegB   = 0x22,      //Copy Wiper B position to Non Volatile Register B
        Copy_Wiper_RegAB_to_NV_RegAB = 0x23,      //Copy both Wiper A & B positions to respective Non Volatile Registers
        Copy_NV_RegA_to_Wiper_RegA   = 0x31,      //Copy Non Volatile Register A to Wiper Register A
        Copy_NV_RegB_to_Wiper_RegB   = 0x32,      //Copy Non Volatile Register B to Wiper Register B
        Copy_NV_RegAB_to_Wiper_RegAB = 0x33      //Copy Non Volatile Register A & B to Wiper Register A & B
    } setting_t;
    
    /**********************************************************//**
    * @brief Constructor for MAX5487 Class.  
    * 
    * @details Requires an existing SPI object as well as a DigitalOut object. 
    * The DigitalOut object is used for a chip enable signal
    *
    * On Entry:
    *     @param[in] spi - pointer to existing SPI object
    *     @param[in] pin - pointer to a DigitalOut pin object
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    MAX5487(SPI &spi, DigitalOut &pin); 
    
    /** 
    * @brief Send write command
    * @param setting - Command sent to MAX5487 register
    * @param value - 8 bit Value to write
    * @return void, output is sent through serial port
    */
    void writeCommand(MAX5487::setting_t setting, int value);
    
    /** 
    * @brief Send copy command
    * @param setting - Command sent to MAX5487 register
    * @return void
    */
    void writeCommand(MAX5487::setting_t setting); 


    /************************************************************
    * @brief Default destructor for MAX5487 Class.  
    *
    * @details Destroys SPI object if owner 
    *
    * On Entry:
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    ~MAX5487();
    private:
    // SPI object
    SPI &m_spi;
    // Selector pin object
    DigitalOut &m_pin;
};

#endif