/* Copyright (c) 2010-2011 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "mbed.h"
#include "FXOS8700Q.h"
#include "FXAS21000.h"
#include "kalman.c"
#include "arm_math.h"

#define PI             3.1415926535897932384626433832795f
#define Rad2Dree       57.295779513082320876798154814105f

#define PID_ROLL_KP        0.0245f              /* Proporcional */ //0.015f
#define PID_ROLL_KI        0.000175f              /* Integral */
#define PID_ROLL_KD        0.0f              /* Derivative */

#define PID_PITCH_KP       0.0245f              /* Proporcional */ //0.015f
#define PID_PITCH_KI       0.000175f              /* Integral */
#define PID_PITCH_KD       0.0f              /* Derivative */

#define ROLL_SP            PI/2
#define PITCH_SP           PI/2

DigitalOut red(LED_RED);
DigitalOut green(LED_GREEN);

FXOS8700Q_acc combo_acc(A5, A4, FXOS8700CQ_SLAVE_ADDR0);
FXOS8700Q_mag combo_mag(A5, A4, FXOS8700CQ_SLAVE_ADDR0);
FXAS21000 gyro(A5, A4);

Timer GlobalTime;
Timer ProgramTimer;

//PwmOut M1(PTB18);
PwmOut M1(D13);
PwmOut M2(D12);
PwmOut M3(D11);
PwmOut M4(D10);

Serial pc(USBTX, USBRX);
Serial bt(D1,D0);

AnalogIn   ultra(A0);

kalman filter_pitch; 
kalman filter_roll;

float R;
double angle[3];
unsigned long timer;
long loopStartTime;
char i;
char command = ' ';
char sflag = 0;
float high;

float ESC1 = 0.0006f;   //pitch up
float ESC2 = 0.0006f;   //roll  up
float ESC3 = 0.0006f;   //roll  down
float ESC4 = 0.0006f;   //pitch down

float roll_error;
float pitch_error;

float roll;
float pitch;

void bt_callback(void);
void esc_start(void);
void esc_stop(void);
float esc_control(float current, float pid, float rate);

int main(void)
{
    float gyro_data[3];
    MotionSensorDataUnits adata;
    MotionSensorDataUnits mdata;
    
    bt.attach(&bt_callback);
    
    printf("\r\nStarting\r\n\r\n");
    
    arm_pid_instance_f32 RPID;
    arm_pid_instance_f32 PPID;
  
      //Pitch
    PPID.Kp = PID_PITCH_KP/1000.0f;        /* Proporcional */
    PPID.Ki = PID_PITCH_KI/1000.0f;        /* Integral */
    PPID.Kd = PID_PITCH_KD/1000.0f;        /* Derivative */
    
    //Roll
    RPID.Kp = PID_ROLL_KP/1000.0f;        /* Proporcional */
    RPID.Ki = PID_ROLL_KI/1000.0f;        /* Integral */
    RPID.Kd = PID_ROLL_KD/1000.0f;        /* Derivative */
    
    arm_pid_init_f32(&RPID, 1);
    arm_pid_init_f32(&PPID, 1);
    
    red = 0; green= 1;
    GlobalTime.start();
    
    M1.period(0.02f);               //Comparten el mismo timer
    M1.pulsewidth(ESC1);
    M2.pulsewidth(ESC2);
    M3.pulsewidth(ESC3);
    M4.pulsewidth(ESC4);
    
    combo_acc.enable();
    combo_mag.enable();
    printf("FXOS8700 Combo = %X\r\n", combo_acc.whoAmI());
    printf("FXAS21000 Gyro = %X\r\n", gyro.getWhoAmI());
    
    kalman_init(&filter_pitch, R_matrix, Q_Gyro_matrix, Q_Accel_matrix); 
    kalman_init(&filter_roll, R_matrix, Q_Gyro_matrix, Q_Accel_matrix);
    
    esc_start();
    wait(2.0f);
    red = 1; green= 0;
    
    ProgramTimer.start();
    loopStartTime = ProgramTimer.read_us();
    timer = loopStartTime;
    
    while(1) {
        
        high = (float)(ultra.read_u16()*2.75f/512.0f)*2.54f;
        
        combo_acc.getAxis(adata);
        combo_mag.getAxis(mdata);
        gyro.ReadXYZ(gyro_data);
        
        R = sqrt(std::pow(adata.x, 2) + std::pow(adata.y, 2) + std::pow(adata.z, 2));
        
        kalman_predict(&filter_pitch, gyro_data[0],  (ProgramTimer.read_us() - timer)); 
        kalman_update(&filter_pitch, acos(adata.x/R));
        kalman_predict(&filter_roll, gyro_data[1],  (ProgramTimer.read_us() - timer)); 
        kalman_update(&filter_roll, acos(adata.y/R));
        
        angle[0] = kalman_get_angle(&filter_pitch);
        angle[1] = kalman_get_angle(&filter_roll);
        
        if (angle[0]>PI) angle[0] = PI;
        else if (angle[0]<0) angle[0] = 0.0f;
        else angle[0] += 0.0f;
        
        if (angle[1]>PI) angle[1] = PI;
        else if (angle[1]<0) angle[1] = 0.0f;
        else angle[1] += 0.0f;
        
        pitch_error = angle[0] - PITCH_SP;
        roll_error = angle[1] - ROLL_SP;
        
        pitch = arm_pid_f32(&PPID, pitch_error);
        roll = arm_pid_f32(&RPID, roll_error);
        
        timer = ProgramTimer.read_us();
        
        if (!sflag){
            
            ESC1 = esc_control(ESC1,-pitch,0.0f);
            ESC2 = esc_control(ESC2,-roll,0.0f);
            ESC3 = esc_control(ESC3,roll,0.0f);
            ESC4 = esc_control(ESC4,pitch,0.0f);
            
            M1.pulsewidth(ESC1);
            M2.pulsewidth(ESC2); 
            M3.pulsewidth(ESC3); 
            M4.pulsewidth(ESC4);
        }
    
        //printf("FXOS8700 Acc:   X:%6.3f Y:%6.3f Z:%6.3f\r\n", adata.x, adata.y, adata.z);
        //printf("FXOS8700 Mag:   X:%6.2f Y:%6.2f Z:%6.2f\r\n", mdata.x, mdata.y, mdata.z);
        //printf("FXAS21000 Gyro: X:%6.2f Y:%6.2f Z:%6.2f\r\n", gyro_data[0], gyro_data[1], gyro_data[2]);
        bt.printf("Roll Angle X: %.6f   Pitch Angle Y: %.6f \r\n", Rad2Dree * angle[1], Rad2Dree * angle[0]);
        bt.printf("roll = %.6f     pitch = %.6f \r\n",roll,pitch);
        bt.printf("ESC1 = %.6f     ESC4 = %.6f \r\n", ESC1,ESC4);
        bt.printf("ESC2 = %.6f     ESC3 = %.6f \r\n", ESC2,ESC3);
        //printf("dist = %.2f \r\n",dist);
        
        wait(0.02f);
    }
}

void bt_callback(void) {
    // Note: you need to actually read from the serial to clear the RX interrupt
    command = bt.getc();
    //if (command == 'z') bt.printf("start\n\r");
    if (command == 'x') {
        sflag = 1;
        bt.printf("stop\n\r"); 
        esc_stop();
    }
    else if (command == 'w') bt.printf("up\n\r"); 
    else if (command == 's') bt.printf("down\n\r"); 
    else if (command == 'h') bt.printf("dist = %.2f \r\n",high);
    else bt.printf("%c\n\r", command);
}

float esc_control(float current, float pid, float rate){
    if ((current + pid + rate)>0.0014f) return 0.0014f;
    else if ((current + pid + rate)<0.001f) return 0.001f; 
    else return current + pid + rate;
}

void esc_start(void) {
    sflag = 0;
    ESC1 = 0.0006f;
    ESC2 = 0.0006f;
    ESC3 = 0.0006f;
    ESC4 = 0.0006f;
    for (i = 0; i < 4; i++){
        ESC1 += 0.0001f;
        ESC2 += 0.0001f;
        ESC3 += 0.0001f;
        ESC4 += 0.0001f;
        M1.pulsewidth(ESC1);
        M2.pulsewidth(ESC2);
        M3.pulsewidth(ESC3);
        M4.pulsewidth(ESC4);
        wait_ms(1000);
    }
}

void esc_stop(void){
    red = 0; green= 1; sflag = 1;
    while ((ESC1 > 0.0006f)||(ESC2 > 0.0006f)||(ESC3 > 0.0006f)||(ESC4 > 0.0006f)){
        ESC1 -= 0.0001f;
        ESC2 -= 0.0001f;
        ESC3 -= 0.0001f;
        ESC4 -= 0.0001f;
        M1.pulsewidth(ESC1);
        M2.pulsewidth(ESC2);
        M3.pulsewidth(ESC3);
        M4.pulsewidth(ESC4);
        wait_ms(250);   
    }
    if (ESC1 < 0.0006f) ESC1 = 0.0006f;
    if (ESC2 < 0.0006f) ESC2 = 0.0006f;
    if (ESC3 < 0.0006f) ESC3 = 0.0006f;
    if (ESC4 < 0.0006f) ESC4 = 0.0006f;
}