#include "stdint.h"
#include "mbed.h"
#include "icytrx_spi.h"
#include "icytrx_cs2.h"
#include "config_cs2.h"
#include "icytrx_cs551.h"
#include "config_cs551.h"


extern SPI spi;
extern Serial pc;
extern DigitalOut cs;
extern DigitalOut nrst;


icytrx::icytrx(){
    spi_mode=0;
}

void icytrx::init(){
    nrst=0;
    wait(0.01);
    nrst=1;
    wait(0.001);
    read_and_set_version();
}

uint8_t icytrx::get_spi_mode(){
    return (uint8_t)spi_mode;
}

uint8_t icytrx::read_reg(uint8_t addr){
     uint8_t r;
     if(spi_mode==0)
          r=read_reg_17(addr);
     else
          r=read_reg_adv(addr);
     return r;
}    

void icytrx::write_reg(uint8_t addr,uint8_t r){
     if(spi_mode==0)
          write_reg_17(addr,r);
     else
          write_reg_adv(addr,r);
}    

void icytrx::write_burst(uint8_t addr,uint8_t *v,int ll){
     if(spi_mode==0)
          write_burst_17(addr,v,ll);
     else
          write_burst_adv(addr,v,ll);
}    

void icytrx::read_burst(uint8_t addr,uint8_t *v,int ll){
     if(spi_mode==0)
          read_burst_17(addr,v,ll);
     else
          read_burst_adv(addr,v,ll);
}    

uint8_t icytrx::read_reg_17(uint8_t addr){
    uint8_t ra,rb;
    cs = 0;
    spi.write(addr>>1);
    ra=(uint8_t)spi.write((addr&0x1)<<7);
    rb=(uint8_t)spi.write(0x00);
    cs = 1;
    ra=ra<<1;
    ra=ra+(rb>>7);
    return ra;
}

uint8_t icytrx::read_reg_adv(uint8_t addr){
    uint8_t ra;
    cs = 0;
    spi.write(0x00);
    spi.write(addr);
    ra = spi.write(0x00);
    cs = 1;
    return ra;
}

void icytrx::write_reg_17(uint8_t addr,uint8_t value){
    cs = 0;
    spi.write(0x80+(addr>>1));
    spi.write(((addr&0x1)<<7)+(value>>1));
    spi.write((value&0x1)<<7);
    cs = 1;
}

void icytrx::write_reg_adv(uint8_t addr,uint8_t value){
    cs = 0;
    spi.write(0x01);
    spi.write(addr);
    spi.write(value);
    cs = 1;
}

void icytrx::read_burst_adv(uint8_t addr,uint8_t *values,int ll){
    int i;
    cs = 0;
    spi.write(0x00);
    spi.write(addr);
    for(i=0;i<ll;i++){
        values[i]=spi.write(0x00);
    }
    cs = 1;
}

void icytrx::write_burst_adv(uint8_t addr,uint8_t *values,int ll){
    int i;
    cs = 0;
    spi.write(0x01);
    spi.write(addr);
    for(i=0;i<ll;i++){
        spi.write(values[i]);
    }
    cs = 1;
}

void icytrx::read_burst_17(uint8_t addr,uint8_t *values,int ll){
    uint8_t ra,rb,i;
    cs = 0;
    spi.write(addr>>1);
    ra=(uint8_t)spi.write((addr&0x1)<<7);
    for(i=0;i<ll;i++){
        rb=(uint8_t)spi.write(0x00);
        values[i]=(ra<<1)+(rb>>7);
        ra=rb;
    }
    cs = 1;
}

void icytrx::write_burst_17(uint8_t addr,uint8_t *values,int ll){
    int i;
    cs = 0;
    spi.write(0x80+(addr>>1));
    spi.write(((addr&0x1)<<7)+(values[0]>>1));
    for(i=0;i<(ll-1);i++)
        spi.write(((values[i]&0x1)<<7)+(values[i+1]>>1));
    spi.write((values[ll-1]&0x1)<<7);
    cs = 1;
}

void icytrx::set_bits(uint8_t addr,uint8_t mask,uint8_t value){
    write_reg(addr,(read_reg(addr)&mask)|value);
}

void icytrx::set_adv_spi(){
    write_reg(REG_SERIAL_CONF,0x1);
    spi_mode=1;
}

void icytrx::soft_rst(){
    write_reg(REG_MODE,0x0);
    write_reg(REG_MODE,0x22);
}

void icytrx::activate_tx(){
    write_reg(REG_FSM_MODE,0x7);
}

void icytrx::activate_rx(){
    write_reg(REG_FSM_MODE,0x3);
}

void icytrx::stop(){
    write_reg(REG_FSM_MODE,0x8);
}

int icytrx::get_fei(){
    uint8_t fei[2];
    int feil,feih;
    read_burst(REG_FEI_PKT,fei,2);
    feil = (int)fei[0];
    feih = (int)fei[1];
    feil+=feih<<8;
    if(feil>32767)
        feil-=65536;
    return feil;
}

uint8_t icytrx::get_rssi(){
    return read_reg(REG_RSSI_PKT);
}

uint8_t icytrx::get_agc(){
    return (read_reg(REG_RX_ATT_LEVEL)>>4)&0xf;
}

void icytrx::set_channel(uint8_t chn){
    if(version>=0x30){
        if(chn==1000)
            write_reg(REG_CHANNEL,chn+0x40);
        else
            write_reg(REG_CHANNEL,chn);
    }
    else{
        double f;
        uint32_t v;
        uint8_t vb[4];
        f=(2.402e3+chn*2.0e3)/144.0*(1<<21);
        v=(uint32_t)f;
        vb[0]=v&0xff;
        vb[1]=(v>>8)&0xff;
        vb[2]=(v>>16)&0xff;
        vb[3]=0x80+((v>>24)&0xff);
        write_burst(REG_CENTER_FREQ,vb,4);
    }
}

void icytrx::set_IRQs(uint8_t txstop,uint8_t rxstop,uint8_t txfifo){
    uint8_t mask;
    if(version<0x30){
        txstop=txstop%6;
        rxstop=rxstop%6;
        txfifo=txfifo%6;
    }
    mask = 0x0f;
    if((txstop&0x1)==1)
      mask = 0xf0;
    set_bits(REG_PAD_CONF_1+(txstop>>1),~mask,mask&0x11);
    mask = 0x0f;
    if((rxstop&0x1)==1)
      mask = 0xf0;
    set_bits(REG_PAD_CONF_1+(rxstop>>1),~mask,mask&0x22);
    mask = 0x0f;
    if((txfifo&0x1)==1)
      mask = 0xf0;
    set_bits(REG_PAD_CONF_1+(txfifo>>1),~mask,mask&0x55);
}
        
void icytrx::read_and_set_version(){
    uint8_t ver;
    ver=read_reg(0xff);
    version=ver;
    switch(ver){
    case 0x31:
        conf_1Mbps      = conf_cs2_1Mbps;
        conf_1Mbps_len  = conf_cs2_1Mbps_len;
        conf_2Mbps      = conf_cs2_2Mbps;
        conf_2Mbps_len  = conf_cs2_2Mbps_len;
        config_version  = (char *)config_version_cs2;
        chip_name       = (char *)chip_name_cs2;
        REG_MODE        = ICYTRX65CS2_MODE_ADDR;
        REG_BANK        = ICYTRX65CS2_BANK_ADDR;
        REG_CHANNEL     = ICYTRX65CS2_CHANNEL_ADDR;
        REG_FSM_MODE    = ICYTRX65CS2_FSM_MODE_ADDR;
        REG_SERIAL_CONF = ICYTRX65CS2_SERIAL_CONF_ADDR;
        REG_CHIP_ID     = ICYTRX65CS2_CHIP_ID_ADDR;
        REG_RSSI_PKT    = ICYTRX65CS2_RSSI_PKT_ADDR;
        REG_FEI_PKT     = ICYTRX65CS2_FEI_PKT_ADDR;
        REG_DESER_STATUS= ICYTRX65CS2_DESER_STATUS_ADDR;
        REG_IRQ_STATUS  = ICYTRX65CS2_IRQ_STATUS_ADDR;
        REG_TXFIFO      = ICYTRX65CS2_TXFIFO_ADDR;
        REG_TX_MAC_TIMER= ICYTRX65CS2_TX_MAC_TIMER_ADDR;
        REG_FIFO        = ICYTRX65CS2_FIFO_ADDR;
        REG_PAD_CONF_1  = ICYTRX65CS2_PAD_CONF_1_ADDR;
        REG_CENTER_FREQ = ICYTRX65CS2_CENTER_FREQ_ADDR;
        REG_IRQ_CONF    = ICYTRX65CS2_IRQ_CONF_ADDR;
        REG_RX_ATT_LEVEL= ICYTRX65CS2_RX_ATT_LEVEL_ADDR;
        break;    
    case 0x50:
    case 0x51:
    case 0x52:
    case 0x53:
    case 0x54:
        conf_1Mbps      = conf_cs551_1Mbps;
        conf_1Mbps_len  = conf_cs551_1Mbps_len;
        conf_2Mbps      = conf_cs551_2Mbps;
        conf_2Mbps_len  = conf_cs551_2Mbps_len;
        config_version  = (char *)config_version_cs551;
        chip_name       = (char *)chip_name_cs551;
        REG_MODE        = ICYTRX55CS1_MODE_ADDR;
        REG_BANK        = ICYTRX55CS1_BANK_ADDR;
        REG_CHANNEL     = ICYTRX55CS1_CHANNEL_ADDR;
        REG_FSM_MODE    = ICYTRX55CS1_FSM_MODE_ADDR;
        REG_SERIAL_CONF = ICYTRX55CS1_SERIAL_CONF_ADDR;
        REG_CHIP_ID     = ICYTRX55CS1_CHIP_ID_ADDR;
        REG_RSSI_PKT    = ICYTRX55CS1_RSSI_PKT_ADDR;
        REG_FEI_PKT     = ICYTRX55CS1_FEI_PKT_ADDR;
        REG_DESER_STATUS= ICYTRX55CS1_DESER_STATUS_ADDR;
        REG_IRQ_STATUS  = ICYTRX55CS1_IRQ_STATUS_ADDR;
        REG_TXFIFO      = ICYTRX55CS1_TXFIFO_ADDR;
        REG_TX_MAC_TIMER= ICYTRX55CS1_TX_MAC_TIMER_ADDR;
        REG_FIFO        = ICYTRX55CS1_FIFO_ADDR;
        REG_PAD_CONF_1  = ICYTRX55CS1_PAD_CONF_1_ADDR;
        REG_CENTER_FREQ = ICYTRX55CS1_CENTER_FREQ_ADDR;
        REG_IRQ_CONF    = ICYTRX55CS1_IRQ_CONF_ADDR;
        REG_RX_ATT_LEVEL= ICYTRX55CS1_RX_ATT_LEVEL_ADDR;
        break;
    default:
        pc.printf("Unknown version 0x%x!\n",ver);
        version=0x0;
        break;
    }        
}

