/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "iBeaconService.h"

#define BLE_CHECK(X)  (X == BLE_ERROR_NONE) ? (printf("{{success}}\r\n")) : printf("{{failure}} %s at line %u ERROR CODE: %u\r\n", #X, __LINE__, (X));
#define BLE_EQUAL(X,Y) ((X)==(Y)) ? (printf("{{sucess}}\n")) : printf("{{failure}}\n");
    
/**
 * For this demo application, populate the beacon advertisement payload
 * with 2 AD structures: FLAG and MSD (manufacturer specific data).
 *
 * Reference:
 *  Bluetooth Core Specification 4.0 (Vol. 3), Part C, Section 11, 18
 */

BLE ble;
Ticker tick;
DigitalOut myled(LED1);
/**
 * The Beacon payload has the following composition:
 * 128-Bit / 16byte UUID = E2 0A 39 F4 73 F5 4B C4 A1 2F 17 D1 AD 07 A9 61
 * Major/Minor  = 0x1122 / 0x3344
 * Tx Power     = 0xC8 = 200, 2's compliment is 256-200 = (-56dB)
 *
 * Note: please remember to calibrate your beacons
 * TX Power for more accurate results.
 */
const uint8_t uuid[] = {0xE2, 0x0A, 0x39, 0xF4, 0x73, 0xF5, 0x4B, 0xC4,
                        0xA1, 0x2F, 0x17, 0xD1, 0xAD, 0x07, 0xA9, 0x61
                       };
uint16_t majorNumber = 1122;
uint16_t minorNumber = 3344;
uint16_t txPower = 0xC8;

Gap::AddressType_t addressType;
Gap::Address_t address;
Gap::Address_t addrTest = {110,100,100,100,100,100};
const static char     DEVICE_NAME[]        = "IBEACONTEST";
Gap::GapState_t state;

void changeAdvPay(void);
void changeScanRes(void);
/**
* Test to change advertisement interval
*/
void changeAdvTime(void){
    tick.detach();
    BLE_CHECK(ble.gap().stopAdvertising());
    ble.gap().setAdvertisingTimeout(0);
    ble.gap().setAdvertisingInterval(500);
    BLE_CHECK(ble.gap().startAdvertising());
}
/**
* Test to change advertisement payload
*/
void changeAdvPay(void){
    tick.detach();
    BLE_CHECK(ble.gap().stopAdvertising());
    ble.gap().clearAdvertisingPayload();
    ble.gap().setAdvertisingTimeout(0);
    BLE_CHECK(ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::LE_GENERAL_DISCOVERABLE | GapAdvertisingData::BREDR_NOT_SUPPORTED));
    BLE_CHECK(ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::OUTDOOR_GENERIC));
    BLE_CHECK(ble.gap().accumulateAdvertisingPayloadTxPower(10));
    uint8_t data[5] = {123,123,123,123,123};
    BLE_CHECK(ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SERVICE_DATA, data, 5));
    //BLE_CHECK(ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME)));
    ble.gap().setAdvertisingInterval(500);
    BLE_CHECK(ble.gap().startAdvertising());
}
/**
* Test to change add a scan response
*/
void changeScanRes(void){
    tick.detach();
    BLE_CHECK(ble.gap().stopAdvertising());
    ble.gap().clearAdvertisingPayload();
    ble.gap().clearScanResponse();
    ble.gap().setAdvertisingTimeout(0);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_SCANNABLE_UNDIRECTED);
    uint8_t data2[5] = {50,50,50,50,50};
    BLE_CHECK(ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SERVICE_DATA, data2, sizeof(data2)));
    uint8_t data[5] = {50,50,50,50,50};
    BLE_CHECK(ble.gap().accumulateScanResponse(GapAdvertisingData::SERVICE_DATA, data, sizeof(data))); 
    ble.gap().setAdvertisingInterval(500);
    BLE_CHECK(ble.gap().startAdvertising());
}
/**
* Test to change advertisement timeout
*/
void timeoutTest(void){
    BLE_CHECK(ble.gap().stopAdvertising());
    ble.gap().clearAdvertisingPayload();
    ble.gap().clearScanResponse();
    ble.gap().setAdvertisingTimeout(5);
    BLE_CHECK(ble.gap().startAdvertising());
}
/**
* Reset function run after every test
*/
void reset(void){
    tick.detach();
    ble.gap().stopAdvertising();
    ble.gap().clearAdvertisingPayload();
    ble.gap().clearScanResponse();
    ble.gap().setAdvertisingTimeout(0);
    ble.gap().setAdvertisingInterval(1000);
    uint8_t data2[5] = {0,0,0,0,0};
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::SERVICE_DATA, data2, sizeof(data2));
    BLE_CHECK(ble.gap().startAdvertising());
}
/**
* Controls which tests are run from input from PC
*/
void testCentre(void){
    int x;
    scanf("%d" , &x); 
    printf("%d:%d:%d:%d:%d:%d\n", address[0], address[1], address[2], address[3], address[4], address[5]);
    GapAdvertisingData testData = ble.gap().getAdvertisingPayload();
    state = ble.gap().getState();
    !(testData.getPayload()) ? printf("{{failure}} ble.gap().getAdvertisingPayload() at line %u\r\n", __LINE__) : printf("{{success}}\r\n");
    (state.connected == 1 || state.advertising == 0) ? printf("{{failure}} ble.gap().getState() at line %u\r\n", __LINE__) : printf("{{success}}\r\n");
    while(true){
        scanf("%d", &x);
        switch (x){
            case 0:
                changeAdvTime();
                break;
            case 1:
                changeAdvPay();
                break;
            case 2:
                timeoutTest();
                break;
            case 3:
                changeScanRes();
                break;
            default:
                return;
        }
        scanf("%d", &x);
        reset();
    }
}

int main(void)
{
    myled = 0;
    
    BLE_CHECK(ble.init());

    iBeaconService ibeacon(ble, uuid, majorNumber, minorNumber, txPower);

    ble.gap().setAdvertisingInterval(1000); /* 1000ms. */
    ble.gap().setAdvertisingTimeout(0);
    
    BLE_CHECK(ble.gap().setAddress(Gap::ADDR_TYPE_PUBLIC, addrTest));
    BLE_CHECK(ble.gap().getAddress(&addressType, address));
    BLE_CHECK(ble.gap().startAdvertising());
    
    GapAdvertisingData testData = ble.gap().getAdvertisingPayload();
    
    printf("{{success}}" "\n" "{{end}}" "\n"); //tells mbedhtrun to finish
    testCentre();
}
