/** @file main.cpp   */
/** @file main.h 
*   
*   \brief 
* Program main features:
*    - enables communication via ethernet
*    - reads data from GY-80 sensor
*    - reads data from built-in touch sensor
*    - enables configuration of Wiz550io ethernet card via PC serial port
*  
* 
* Peripherals connection details:
*
* WIZ550io pins are connected to:
*    - MOSI  ->  PTD2
*    - MISO  ->  PTD3
*    - SCLK  ->  PTD1
*    - CS    ->  PTD0
*    - RST   ->  PTA20
*    - VCC   ->  +3.3V
*    - GND   ->  GND
*    
* IMU GY-80 pins:
*    - VCC   ->  +5V
*    - GND   ->  GND
*    - SDA   ->  PTC9
*    - SCL   ->  PTC8
*    - Connection to other I2C port may not work!!!
*    
* Serial port communication is available via virtual OpenSDA COM port.
* Detailed description about configuring OpenSDA available here: https://mbed.org/handbook/Windows-serial-configuration .
*
* Digital I/O ports pins and names:
*   - PTE20  ->  P1
*   - PTE21  ->  P2
*   - PTE22  ->  P3
*   - PTE23  ->  P4
*   - PTE29  ->  P5
*   - PTE30  ->  P6
*
* Ethernet commands:
* Available commands:
*   - "read_sensor" - reads all sensor data - Reply form: "Sensor data: magnX, magnY, magnZ, accelX, accelY, accelZ, gyroX, gyroY, gyroZ, TSIposition"
*   - "check_ports" - reads ports P1 to P6 status - Reply form: "Port values: P1: %d, P2: %d, P3: %d, P4: %d, P5: %d, P6: %d"
* Single port commands:
*   - "set_P$_i" - sets port as input    - Reply: "P1 is now set as input"
*   - "set_P$_o" - sets port as output   - Reply: "P1 is now set as output"
*   - "set_P$_1" - sets port as 1 (High) - Reply: "P1 value is now: %d "
*   - "set_P$_0" - sets port as 0 (low)  - Reply: "P1 value is now: %d " 
*   - "get_P$"   - gets port value       - Reply: "P1 value: %d"
*   -  $ mark is port number from 1-6 
*
*   @author Mateusz Jaskula
*
*/



// Set server communication port to 22
#define ECHO_SERVER_PORT   22


/// SPI communication with WIZ550io initialization
SPI spi(PTD2, PTD3, PTD1); /// mosi, miso, sclk

/// Ethernet communication via WIZ550io
WIZnetInterface eth(&spi, PTD0, PTA20); /// spi, cs, reset

/// Serial communication init
Serial pc(USBTX,USBRX);

/// TSI Electrodes definition:
#define ELEC0 9
/// TSI Electrodes definition:
#define ELEC1 10


/// port P1 is connected to PTE20 pin
DigitalInOut P1(PTE20);                         
/// port P1 is connected to PTE21 pin
DigitalInOut P2(PTE21);                         
/// port P2 is connected to PTE22 pin
DigitalInOut P3(PTE22);                         
/// port P3 is connected to PTE23 pin
DigitalInOut P4(PTE23);                         
/// port P4 is connected to PTE29 pin
DigitalInOut P5(PTE29);                         
/// port P5 is connected to PTE30 pin
DigitalInOut P6(PTE30);                         


/// Defines use of DHCP protocol or Static IP Configuration
bool USE_DHCP = true;

// Static IP setting. Using MAC stored in WIZ550io
 char IP_Addr[16] = "192.168.133.015";          /// IP
 char IP_Subnet[16]  = "255.255.255.000";       /// Subnet Mask
 char IP_Gateway[16] = "192.168.133.001";       /// Gateway

/// MAC Adress global variable
uint8_t mac[6];

/// GY80 IMU sensor Object
GY80 sensor;

// GY80 Data is being hold here
float  magn[3];     /// Current magnitude
float accel[3];     /// Current acceleration
float  gyro[3];     /// current gyroscope

/// TSI Slider object:
TSIAnalogSlider tsi(ELEC0, ELEC1, 100);

/// TSI Slider position:
float TSI_pos;

/// Ticker generates time based interrups to read data from sensors in 10ms intervals.
Ticker tick1;


//Functions declaration:
// Initializes eth communication with MAC, IP, Subnet Mask asd Gateway
int init_eth();
// Handles ethernet communication
void eth_comm();
// Reads data from sensor with 1ms freq
void read_data();
// Reads commands and takes action
void comm_handler(char* command);
// Enables configuration mode
void SerialInterHandler();
// 5 sec serial menu with info how to enter configuration mode
void menu_prop();
