/* Program demonstrating data logging to SD card 
    Temperature is read in from an TMP36 
    The time since program start and the voltage 
    are written to a tab-delimited data file. 
    TMP36 pins: left = power, middle = output (p18), right = ground 
    SD pins: DI = p5, DO = p6, SCK = p7, CS = p8  
*/ 
#include "mbed.h" 
#include "SDFileSystem.h" 

AnalogIn tmp36(p18); 
Serial pc(USBTX,USBRX); 
SDFileSystem fs(p5, p6, p7, p8, "fs"); 
Timer t; 
Ticker sampleTime; 
DigitalOut doneLED(LED1); 
bool timeToRead; 

void triggerCollection() { 
    timeToRead = true;  // flag to collect data 
} 

int main() { 

    float voltage, temperature; 
    // Mount the filesystem 
    bool mountFailure = fs.mount(); 
    if (mountFailure != 0) { 
        pc.printf("Failed to mount the SD card.\r\n"); 
        return -1;  // ends program with error status 
    }

    FILE* fp = fopen("/fs/log.txt","w"); 
    if (fp == NULL) { 
        pc.printf("Failed to open the file.\r\n"); 
        fs.unmount(); 
        return -1; 
    }    

    // Write a header row 
    fprintf(fp, "Time (s) \t Temperature (deg C)\r\n"); 

    // Start the timer and ticker 
    t.start(); 
    sampleTime.attach(&triggerCollection, 0.5);  // write data every 0.5 s 
    timeToRead = true; // collect data at t = 0 

    while (t.read()<20) { 
        if (timeToRead) { 
            timeToRead = false;  // reset data collection flag 
            voltage = 3.3*tmp36.read(); 
            temperature = (voltage-0.5)/0.01; 
            fprintf(fp,"%.2f \t %.2f\r\n", t.read(),temperature); 
        } 
    } 
    
    // Close the file and unmount the file system so the SD card is happy 
    fclose(fp); 
    fs.unmount(); 
    // Turn on LED to let user know it is safe to remove the SD card 
    doneLED = 1; 
} 