/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "MicroBit.h"

#include "bme680.h"
#include "sgp30.h"
#include "IaqNonVolatileStore.h"
#include "JavaScriptRadio.h"

#include "physics.h"

#include "MovingAverage.h"
#include "RangeTransform.h"

#include "AppSpecificTestrunner.h"

#define RADIO_GROUP 14
#define RADIO_DELAY 800

#define CMD_SET_BASELINE            "bl "
#define CMD_MEASURE_AND_PUBLISH     "IAQ"
#define CMD_RESET_TEMP_HUM_CO2      "resetTmpHum"
#define CMD_STORE_CURRENT_BASELINE  "updateBaseline"

MicroBit uBit;
Bme680* bme680 = NULL;
struct bme680_field_data* bme680Data = NULL;
Sgp30* sgp30 = NULL;
I2cCallbacks* callbacks = NULL;
IaqNonVolatileStore* nvStore = NULL;
JavaScriptRadio* radio = NULL;

bool cancelMeasureLoops = false;
int runningLoops = 0;
int displayBrightness = 255;

void waitForLooppsToFinish() {
    cancelMeasureLoops=true;
    while (runningLoops>0) {
        uBit.sleep(10);
    }
    cancelMeasureLoops = false;
}


void displayPixels(int ystart, int pixels, int maxpixels) {
    int y = ystart;
    for (int i=0; i+((y-ystart)*5)<maxpixels; ++i) {
        if (i==5) {
            ++y;
            i=0;
        }
        int pixelsused = i+((y-ystart)*5);
        uBit.display.image.setPixelValue(i, y, pixelsused<pixels? displayBrightness : 0);
    }
}

int measureBme680(){
    int result = BME680_E_DEV_NOT_FOUND;
    if (bme680!=NULL && bme680Data!=NULL) {
        result = bme680->measure(
            bme680Data,
            bme680Data->temperature==0 ?
                uBit.thermometer.getTemperature()
                : bme680Data->temperature,
            2000, 350);
    }
    return result;
}

void measureAndDisplayBme680() {
    if (bme680!=NULL) {
        // measuring
        uBit.display.image.setPixelValue(4, 4, displayBrightness);
        if (measureBme680()==MICROBIT_OK) {
            nvStore->updateGas(bme680Data->gas_resistance);
            nvStore->updateTemp(bme680Data->temperature);
            nvStore->updatePress(bme680Data->pressure);
            nvStore->updateHumidity(bme680Data->humidity);
            
            const uint32_t gasMax = nvStore->getGasMax();
            
            // status (last line: 0 sensor found, 2, stray signals, 4, measuring)
            uBit.display.image.setPixelValue(0, 4, displayBrightness);
            uBit.display.image.setPixelValue(2, 4, nvStore->strayData() || gasMax < bme680Data->gas_resistance ? displayBrightness : 0);
            // will be set below uBit.display.image.setPixelValue(4, 4, 0);
            
            // if sgp 30 exists, we have less room fpr bme680 resukts
            const int bmeY = sgp30 != NULL ? 2 : 0;
            const int bmeMaxPixels = sgp30 != NULL ? 5 : 15;
            
            // bme 680 gas state
            const int bmeGasPixels = (gasMax - bme680Data->gas_resistance) * bmeMaxPixels / gasMax;
            displayPixels(bmeY, bmeGasPixels, bmeMaxPixels);
            
            // humidity index
            displayPixels(3, 5*bme680Data->humidity/100000, 5);
        } else {
            // indicate sensor not working
            uBit.display.image.setPixelValue(0, 4, 0);
        }
        // not measuring
        uBit.display.image.setPixelValue(4, 4, 0);
    }
}

bool checkAndStoreSgp30Baseline() {
    uint16_t eco2_base = 0;
    uint16_t tvoc_base = 0;
    bool baseLineOK = sgp30->getIAQBaseline(&eco2_base, &tvoc_base);
    if (baseLineOK) {
        nvStore->storeIAQBaseline(eco2_base, tvoc_base);
        /*radio -> sendToMakeCodeDevices("sgp30 baseln upd");
        radio -> sendToMakeCodeDevices(ManagedString(eco2_base)
            + " / "
            + ManagedString(tvoc_base));
        if (nvStore->getIAQBaseline(&eco2_base, &tvoc_base)) {
            radio -> sendToMakeCodeDevices(ManagedString(eco2_base)
             + " / "
                + ManagedString(tvoc_base));
        }*/
    }
    
    return baseLineOK;
}

void measureAndDisplaySgp30() {
    if (sgp30!=NULL) {
        int sgpMaxPixels = 10;
        int secondLineStart = 2;
        if (bme680!=NULL) {
            sgpMaxPixels = 5;
            secondLineStart = 1;
            uBit.display.image.setPixelValue(3, 4, displayBrightness);
            if (sgp30->setHumidity(bme680Data->humidity, bme680Data->temperature)) {
                uBit.display.image.setPixelValue(3, 4, 0);
            }
            uBit.sleep(10);
        }
        uBit.display.image.setPixelValue(3, 4, displayBrightness);
        bool measureOK = sgp30->IAQmeasure();
        if (measureOK) {
            sgp30->IAQmeasureRaw();
            uBit.display.image.setPixelValue(1, 4, displayBrightness);
            uBit.display.image.setPixelValue(3, 4, 0);
            
            if (sgp30->TVOC < 60000) nvStore->updateVoc(sgp30->TVOC);
            if (sgp30->eCO2 < 60000) nvStore->updateCo(sgp30->eCO2);
            
            int co2Dots = min (5, sgp30->eCO2 /1500);
            displayPixels(0, 5 - RangeTransform::exponentialTransform(sgp30->TVOC, 20000, sgpMaxPixels), sgpMaxPixels);
            displayPixels(secondLineStart, co2Dots, sgpMaxPixels);
            checkAndStoreSgp30Baseline();
        } else {
            uBit.display.image.setPixelValue(1, 4, 0);
        }
    }
}


void bmeMeasureLoop() { 
    if (bme680!=NULL) {
        ++runningLoops;
        while (!cancelMeasureLoops){
            measureAndDisplayBme680();
            uBit.sleep(500);
        }
        --runningLoops;
    }
    release_fiber();
}

void sgpMeasureLoop() {
    if (sgp30!=NULL) {
         ++runningLoops;
        while (!cancelMeasureLoops){
            measureAndDisplaySgp30();
            uBit.sleep(950);
        }
        --runningLoops;
    }
    release_fiber();
}

void startMeasureLoops() {
    if (runningLoops>0) {
        uBit.display.scroll("already running");
        return;
    }
    create_fiber(sgpMeasureLoop);
    create_fiber(bmeMeasureLoop);
}


void init680(){
    if (bme680!=NULL){
        delete bme680;
    }
    
    uint32_t gasMax = nvStore->getGasMax();
    
    bme680 = new Bme680(callbacks);
    int code = bme680->init();
    if (code == MICROBIT_OK){
        if (bme680Data==NULL) {
            bme680Data = new struct bme680_field_data;
        }
        code = bme680->measure(
            bme680Data,
            uBit.thermometer.getTemperature(),
            100, 100);
    }
    if (code != MICROBIT_OK){
        delete bme680;
        bme680 = NULL;
        delete bme680Data;
        bme680Data = NULL;
        uBit.display.scroll(code);
        ManagedString send("no BME680 " + code);
        radio -> sendToMakeCodeDevices(send);
    } else {
        uBit.display.image.setPixelValue(0, 4, displayBrightness);
        //radio -> sendToMakeCodeDevices("BME680 up");
    }
}

void initSgp30(){
    if (sgp30!=NULL){
        delete sgp30;
    }
    
    sgp30 = new Sgp30(callbacks);
    if (sgp30->test() && sgp30->begin()) {
        uBit.display.image.setPixelValue(1, 4, displayBrightness);
        //radio -> sendToMakeCodeDevices("sgp30 up");
        
        uint16_t eco2_base;
        uint16_t tvoc_base;
        if (nvStore->getIAQBaseline(&eco2_base, &tvoc_base)) {
            sgp30->setIAQBaseline(eco2_base, tvoc_base);
            uBit.display.image.setPixelValue(2, 4, displayBrightness);
            /*
            uBit.sleep(RADIO_DELAY);
            radio -> sendToMakeCodeDevices("sgp30 basline");
            uBit.sleep(RADIO_DELAY);
            radio -> sendToMakeCodeDevices("eco: "
                + ManagedString(eco2_base));
            uBit.sleep(RADIO_DELAY);
            radio -> sendToMakeCodeDevices("co2: "
                + ManagedString(tvoc_base)
                );
            */
        } else {
            uBit.sleep(RADIO_DELAY);
            //radio -> sendToMakeCodeDevices("sgp30 no baseline");
        }
    } else  {
        radio -> sendToMakeCodeDevices("no sgp30");
        delete sgp30;
        sgp30 = NULL;
    }
}

void sgp30SetBaseline(ManagedString message){
    // sgp30->setIAQBaseline(eco2_base, tvoc_base);
    if (! (message.substring(0, 3) == CMD_SET_BASELINE)) {
        radio -> sendToMakeCodeDevices("sgp30 not setting baseline");
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices(message);
        return;
    }
    
    uint16_t eco2_base = 0;
    uint16_t tvoc_base = 0;
    
    int messageLength = message.length();
    
    int i = 3;
    char c = ' ';
    
    do{
        c = message.charAt(i);
        if (c>='0' && c<='9') {
            eco2_base*=10;
            eco2_base+= c-'0';
        }
    } while (++i < messageLength && c != ' ');
    
    if (eco2_base!=0 && c==' ') {
        do{
            c = message.charAt(i);
            
            if (c>='0' && c<='9') {
                tvoc_base*=10;
                tvoc_base+= c-'0';
            }
        } while (++i < messageLength);
    }
    
    if (tvoc_base != 0){
        sgp30->setIAQBaseline(eco2_base, tvoc_base);
        nvStore->storeIAQBaseline(eco2_base, tvoc_base);
    
        radio -> sendToMakeCodeDevices("sgp30 baseline set and stored");
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices(message);
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices("eco: " + ManagedString(eco2_base));
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices("voc: " + ManagedString(tvoc_base));
    } else {
        radio -> sendToMakeCodeDevices("sgp30 not setting baseline");
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices(message);
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices("eco:");
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices(eco2_base);
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices("voc:");
        uBit.sleep(RADIO_DELAY);
        radio -> sendToMakeCodeDevices(tvoc_base);
    }
    
}

void initSensors() {
    waitForLooppsToFinish();
    init680();
    initSgp30();    
}

void publishResults() {
    waitForLooppsToFinish();
    if (sgp30 != NULL) {
        ManagedString send = ManagedString("SgpTvoc ")
            + ManagedString((int)sgp30->TVOC)
            + " "
            + ManagedString((int)nvStore->getVocMax());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        //send = ManagedString("SgpMaxVoc ") + ManagedString((int)nvStore->getVocMax());
        //radio -> sendToMakeCodeDevices(send);
        //uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("SgpO2 ")
            + ManagedString((int)sgp30->eCO2)
            + " "
            + ManagedString((int)nvStore->getCoMax());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        //send = ManagedString("SgpMaxCo2 ") + ManagedString((int)nvStore->getCoMax());
        //radio -> sendToMakeCodeDevices(send);
        //uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("SgpRawH ")
            + ManagedString((int)sgp30->rawH2);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        send = ManagedString("SgpRawE ")
            + ManagedString((int)sgp30->rawEthanol);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
    }else {
         radio -> sendToMakeCodeDevices("IAQ no sgp30");
         uBit.sleep(RADIO_DELAY);
    }
    
    if (bme680Data!=NULL) {
        ManagedString send = ManagedString("Bme680G ") + ManagedString((int)bme680Data->gas_resistance);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680+G ") + ManagedString((int)nvStore->getGasMax());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680-G ") + ManagedString((int)nvStore->getGasMin());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
          
        send = ManagedString("Bme680T ") + ManagedString((int)bme680Data->temperature);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680+-T ")
            + ManagedString((int)nvStore->getTempMax())
            + ManagedString(" ")
            + ManagedString((int)nvStore->getTempMin());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        /*
        send = ManagedString("Bme680-T ") + ManagedString((int)nvStore->getTempMin());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        */
        send = ManagedString("Bme680Press ") + ManagedString((int)bme680Data->pressure);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680+P ") + ManagedString((int)nvStore->getPressMax());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        send = ManagedString("Bme680-P ") + ManagedString((int)nvStore->getPressMin());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680Hum ") + ManagedString((int)bme680Data->humidity);
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        
        send = ManagedString("Bme680+H ") + ManagedString((int)nvStore->getHumMax());
        radio -> sendToMakeCodeDevices(send);
        uBit.sleep(RADIO_DELAY);
        send = ManagedString("Bme680-H ") + ManagedString((int)nvStore->getHumMin());
        radio -> sendToMakeCodeDevices(send);
     } else {
         radio -> sendToMakeCodeDevices("IAQ no BME680");
    }
    startMeasureLoops();
    release_fiber();
}

/*
void onButtonA(MicroBitEvent evt)
{
    if (runningLoops>0) {
        displayValuesTxt();
    } else {
        startMeasureLoops();
    }
}
*/
void switchLightLevel(MicroBitEvent evt)
{
    switch (displayBrightness) {
        case 255:
            displayBrightness = 1;
            break;
        case 1:
            displayBrightness =0;
            break;
        default:
            displayBrightness = 255;
    }
}


void clearIQQBaseline(MicroBitEvent evt)
{
    nvStore->clearIQQBaseline();
    uBit.display.scroll("base line clear");
    uBit.reset();
}


const char* runSofwareModuleTests() {
    // heap we've got plenty, stack is rare
    AppSpecificTestrunner* runner = new AppSpecificTestrunner(&uBit);
    const char* result = runner->runAll();
    delete runner;
    return result;
    }

void onData(MicroBitEvent e) {
    ManagedString rcv= radio->received();
    if (rcv == CMD_MEASURE_AND_PUBLISH) {
        create_fiber(publishResults);
    } else if (rcv == CMD_RESET_TEMP_HUM_CO2) {
        nvStore->resetTmpHumCo2();
        create_fiber(publishResults);
    } else if (rcv == CMD_STORE_CURRENT_BASELINE) {
        if (!checkAndStoreSgp30Baseline()) {
            radio -> sendToMakeCodeDevices("sgp30 baseln not upd");
        }
    } else if (rcv.substring(0, 3) == CMD_SET_BASELINE) {
        sgp30SetBaseline(rcv);
    }
}

int main()
{
    uBit.init();
    uBit.display.scroll("i");
    
    //uBit.messageBus.listen(MICROBIT_ID_BUTTON_A, MICROBIT_BUTTON_EVT_CLICK, onButtonA);
    uBit.messageBus.listen(MICROBIT_ID_BUTTON_B, MICROBIT_BUTTON_EVT_CLICK, switchLightLevel);
    uBit.messageBus.listen(MICROBIT_ID_BUTTON_AB, MICROBIT_BUTTON_EVT_CLICK, clearIQQBaseline);
    uBit.messageBus.listen(MICROBIT_ID_RADIO, MICROBIT_RADIO_EVT_DATAGRAM, onData);
    
    callbacks = new I2cCallbacks(&uBit);
    nvStore = new IaqNonVolatileStore(&uBit);
    
    radio = new JavaScriptRadio(&uBit, RADIO_GROUP);

    uBit.display.scroll("t");
    const char* testResults = runSofwareModuleTests();
    if (! Testrunner::messageOK(testResults)) {
        uBit.display.scroll(testResults);
        //radio -> sendToMakeCodeDevices("IAQ tests failed " +ManagedString(testResults));
        return -1;
    }
    
    uBit.display.scroll("s");
    // includes hardware tests
    initSensors();
    
    startMeasureLoops();
    
    release_fiber();
}