#include "IaqNonVolatileStore.h"

#include "MovingAverage.h"

/* from microbit.storage:
    Key Value pairs have a fixed length key of 16 bytes, and a fixed length value of 32 bytes.
    This class only populates a single block (1024 bytes) in its current state,
    which means that 21 Key Value pairs can be stored.
*/

#define GAS_MAX_STORE_KEY "bme_max_res" // 1
#define GAS_MIN_STORE_KEY "bme_min_res" // 2
#define TEMP_MAX_STORE_KEY "temp_max"   // 3
#define TEMP_MIN_STORE_KEY "temp_min"   // 4
#define PRESS_MAX_STORE_KEY "press_max" // 5
#define PRESS_MIN_STORE_KEY "press_min" // 6
#define HUM_MAX_STORE_KEY "hum_max"     // 7
#define HUM_MIN_STORE_KEY "hum_min"     // 8
#define VOC_MAX_STORE_KEY "voc_max"     // 9
#define CO_MAX_STORE_KEY "co_max"       // 10
#define SGP30_ECO_BASE_KEY "eco2_base"  // 11
#define SGP30_VOC_BASE_KEY "tvoc_base"  // 12


const int IaqNonVolatileStore::AVERAGE_BUFFER_SIZE = 10;

IaqNonVolatileStore::IaqNonVolatileStore(MicroBit* uBit) {
    _uBit = uBit;
    _maxGas = getStored(GAS_MAX_STORE_KEY, 0);
    _minGas = getStored(GAS_MIN_STORE_KEY, 0xffffffff);
    
    _tempMax = getStored16(TEMP_MAX_STORE_KEY, 0xffff);
    _tempMin = getStored16(TEMP_MIN_STORE_KEY, 0x7fff);
    
    _pressMax = getStored(PRESS_MAX_STORE_KEY, 0);
    _pressMin = getStored(PRESS_MIN_STORE_KEY, 0xffffffff);
    
    _humMax = getStored(HUM_MAX_STORE_KEY, 0);
    _humMin = getStored(HUM_MIN_STORE_KEY, 0xffffffff);
    
    _vocMax = getStored(VOC_MAX_STORE_KEY, 0);
    
    _coMax = getStored(CO_MAX_STORE_KEY, 0);
    
    _movingAverage = new MovingAverage(AVERAGE_BUFFER_SIZE);
}

IaqNonVolatileStore::~IaqNonVolatileStore() {
    if (_movingAverage!=NULL) delete _movingAverage;
    }

void IaqNonVolatileStore::clear() {
    _uBit->storage.remove(GAS_MAX_STORE_KEY);
    _uBit->storage.remove(SGP30_ECO_BASE_KEY);
    _uBit->storage.remove(SGP30_VOC_BASE_KEY);
    resetTmpHumCo2();
}

void IaqNonVolatileStore::resetTmpHumCo2() {
    _uBit->storage.remove(TEMP_MAX_STORE_KEY);
    _uBit->storage.remove(TEMP_MIN_STORE_KEY);
    _uBit->storage.remove(HUM_MAX_STORE_KEY);
    _uBit->storage.remove(HUM_MIN_STORE_KEY);
    _uBit->storage.remove(GAS_MIN_STORE_KEY);
    _uBit->storage.remove(PRESS_MAX_STORE_KEY);
    _uBit->storage.remove(PRESS_MIN_STORE_KEY);
    _uBit->storage.remove(CO_MAX_STORE_KEY);
    _uBit->storage.remove(VOC_MAX_STORE_KEY);
    
    _tempMax = 0xffff;
    _tempMin = 0x7fff;
    _minGas  = 0xffffffff;
    _humMax = 0;
    _humMin = 0xffffffff;
    _pressMax = 0;
    _pressMin = 0xffffffff;
    
    _coMax = 0;
    _vocMax = 0;
}

uint32_t IaqNonVolatileStore::getGasMax() {
    return _maxGas;
}

uint32_t IaqNonVolatileStore::getGasMin(){
    return _minGas;
}

void IaqNonVolatileStore::updateGas(uint32_t gas) {
    update(gas, &_maxGas, &_minGas, GAS_MAX_STORE_KEY, GAS_MIN_STORE_KEY);
}

void IaqNonVolatileStore::updatePress(const uint32_t pressure) {
    update(pressure, &_pressMax, &_pressMin, PRESS_MAX_STORE_KEY, PRESS_MIN_STORE_KEY);
}
void IaqNonVolatileStore::updateHumidity(const uint32_t humidity) {
    update(humidity, &_humMax, &_humMin, HUM_MAX_STORE_KEY, HUM_MIN_STORE_KEY);
}

int16_t IaqNonVolatileStore::getTempMax(){
    return _tempMax;
}

int16_t IaqNonVolatileStore::getTempMin(){
    return _tempMin;
}

void IaqNonVolatileStore::storeIAQBaseline(uint16_t eco2_base, uint16_t tvoc_base) {
    _uBit->storage.put(SGP30_ECO_BASE_KEY, (uint8_t *)&eco2_base, sizeof(uint16_t));
    _uBit->storage.put(SGP30_VOC_BASE_KEY, (uint8_t *)&tvoc_base, sizeof(uint16_t));
}

bool IaqNonVolatileStore::getIAQBaseline(uint16_t *eco2_base, uint16_t *tvoc_base) {
    bool ok = false;
    KeyValuePair* storedEcoBase = _uBit->storage.get(SGP30_ECO_BASE_KEY);
    KeyValuePair* storedVocBase = _uBit->storage.get(SGP30_VOC_BASE_KEY);
    if (storedEcoBase!=NULL && storedVocBase!=NULL) {
        memcpy(eco2_base, storedEcoBase->value, sizeof(uint16_t));
        delete storedEcoBase;
        memcpy(tvoc_base, storedVocBase->value, sizeof(uint16_t));
        delete storedVocBase;
        ok = true;
    }
    return ok;
}

void IaqNonVolatileStore::clearIQQBaseline(){
    _uBit->storage.remove(SGP30_ECO_BASE_KEY);
    _uBit->storage.remove(SGP30_VOC_BASE_KEY);
}

void IaqNonVolatileStore::updateTemp(const int16_t temp) {
    if (_tempMax < temp){
         _tempMax = temp;
         _uBit->storage.put(TEMP_MAX_STORE_KEY, (uint8_t *)_tempMax, sizeof(int16_t));
    }
    if (_tempMin > temp) {
        _tempMin = temp;
        _uBit->storage.put(TEMP_MIN_STORE_KEY, (uint8_t *)_tempMin, sizeof(int16_t));
    }
}

uint32_t IaqNonVolatileStore::getPressMax(){
    return _pressMax;
}

uint32_t IaqNonVolatileStore::getPressMin(){
    return _pressMin;
}

uint32_t IaqNonVolatileStore::getHumMax(){
    return _humMax;
}

uint32_t IaqNonVolatileStore::getHumMin(){
    return _humMin;
}

uint32_t IaqNonVolatileStore::getVocMax(){
    return _vocMax;
}

void IaqNonVolatileStore::updateVoc(const uint32_t voc) {
    update(voc, &_vocMax, NULL, VOC_MAX_STORE_KEY, NULL);
}

uint32_t IaqNonVolatileStore::getCoMax(){
    return _coMax;
}

void IaqNonVolatileStore::updateCo(const uint32_t co) {
    update(co, &_coMax, NULL, CO_MAX_STORE_KEY, NULL);
}

void IaqNonVolatileStore::update(const uint32_t value, uint32_t* maxVal, uint32_t* minVal, const char* maxKey, const char* minKey) {
    if (*maxVal < value) {
        *maxVal = value;
        _uBit->storage.put(maxKey, (uint8_t *)maxVal, sizeof(uint32_t));
    }
    if (minVal!= NULL && *minVal>value) {
        *minVal = value;
        _uBit->storage.put(minKey, (uint8_t *)minVal, sizeof(uint32_t));
    }
}

const uint32_t* IaqNonVolatileStore::debugInfo() {
    return _movingAverage->debugInfo();
    }
    
bool IaqNonVolatileStore::strayData() {
    const uint32_t* values = _movingAverage->debugInfo();
    uint32_t max = values[0];
    for (int i=0 ; i<AVERAGE_BUFFER_SIZE; ++i) {
        uint32_t val = values[i];
        if (val > max) {
            max = val;
            }
    }
   
    bool stray = false;
    for (int i=0 ; stray==false && i<AVERAGE_BUFFER_SIZE; ++i) {
        stray = (max - values[i]) > 100000;
        }
    return stray;
    }

uint32_t IaqNonVolatileStore::getStored(const char * key, const uint32_t defaultVal){
    uint32_t value = defaultVal;
    KeyValuePair* storedVal = _uBit->storage.get(key);
    if (storedVal!=NULL) {
        memcpy(&value, storedVal->value, sizeof(uint32_t));
        delete storedVal;
    }
    return value;
}

int16_t IaqNonVolatileStore::getStored16(const char * key, const int16_t defaultVal){
    int16_t value = defaultVal;
    KeyValuePair* storedVal = _uBit->storage.get(key);
    if (storedVal!=NULL) {
        memcpy(&value, storedVal->value, sizeof(int16_t));
        delete storedVal;
    }
    return value;
}