#include "sgp30.h"
#include "physics.h"

// the i2c address
#define SGP30_I2CADDR_DEFAULT 0x58     ///< SGP30 has only one I2C address

// commands and constants
#define SGP30_FEATURESET       0x0020  ///< The required set for this library
#define SGP30_CRC8_POLYNOMIAL  0x31    ///< Seed for SGP30's CRC polynomial
#define SGP30_CRC8_INIT        0xFF    ///< Init value for CRC
#define SGP30_WORD_LEN         2       ///< 2 bytes per word

Sgp30::Sgp30(I2cCallbacks *callbacks) {
    _callbacks = callbacks;
    _i2caddr = SGP30_I2CADDR_DEFAULT <<1;
}

bool Sgp30::begin() {
  uint8_t command[2];
  command[0] = 0x36;
  command[1] = 0x82;
  if (!readWordFromCommand(command, 2, 10, serialnumber, 3)) {
    return false;
  }
  
  uint16_t featureset;
  command[0] = 0x20;
  command[1] = 0x2F;
  if (! readWordFromCommand(command, 2, 10, &featureset, 1)) {
    return false;
  }
  
  //Serial.print("Featureset 0x"); Serial.println(featureset, HEX);
  if (featureset != SGP30_FEATURESET)  {
    return false;
  }
  
  if (! IAQinit()) { 
    return false;
  }
  
  return true;
}

bool Sgp30::IAQmeasure() {
    uint8_t command[2];
    command[0] = 0x20;
    command[1] = 0x08;
    uint16_t reply[2];
    if (! readWordFromCommand(command, 2, 12, reply, 2)) {
        return false;
    }
    TVOC = reply[1];
    eCO2 = reply[0];
    return true;
}

bool Sgp30::IAQmeasureRaw(void) {
  uint8_t command[2];
  command[0] = 0x20;
  command[1] = 0x50;
  uint16_t reply[2];
  if (!readWordFromCommand(command, 2, 25, reply, 2))
    return false;
  rawEthanol = reply[1];
  rawH2 = reply[0];
  return true;
}

bool Sgp30::test() {
    uint8_t command[2];
    command[0] = 0x20;
    command[1] = 0x32;
    uint16_t reply;
    if (! readWordFromCommand(command, 2, 220, &reply, 1)) {
        return false;
    }
    
    return reply==0xD400;
}

bool Sgp30::getIAQBaseline(uint16_t *eco2_base,
                                       uint16_t *tvoc_base) {
  uint8_t command[2];
  command[0] = 0x20;
  command[1] = 0x15;
  uint16_t reply[2];
  if (!readWordFromCommand(command, 2, 10, reply, 2))
    return false;
  *eco2_base = reply[0];
  *tvoc_base = reply[1];
  return true;
}

bool Sgp30::setIAQBaseline(uint16_t eco2_base, uint16_t tvoc_base) {
  uint8_t command[8];
  command[0] = 0x20;
  command[1] = 0x1e;
  command[2] = tvoc_base >> 8;
  command[3] = tvoc_base & 0xFF;
  command[4] = generateCRC(command + 2, 2);
  command[5] = eco2_base >> 8;
  command[6] = eco2_base & 0xFF;
  command[7] = generateCRC(command + 5, 2);

  return readWordFromCommand(command, 8, 10);
}

bool Sgp30::setHumidity(const uint32_t rel_humidity, const uint32_t temperature) {
    return setHumidity(Physics::absHumidity(rel_humidity, temperature));
}

bool Sgp30::IAQinit() {
  uint8_t command[2];
  command[0] = 0x20;
  command[1] = 0x03;
  return readWordFromCommand(command, 2, 10);
}

bool Sgp30::setHumidity(const uint32_t absolute_humidity) {
  if (absolute_humidity > 256000) {
    return false;
  }
  
  if (absolute_humidity == _absolute_humidity) {
      return true;
  }

  uint16_t ah_scaled = (uint16_t)(((uint64_t)absolute_humidity * 256 * 16777) >> 24);
  uint8_t command[5];
  command[0] = 0x20;
  command[1] = 0x61;
  command[2] = ah_scaled >> 8;
  command[3] = ah_scaled & 0xFF;
  command[4] = generateCRC(command+2, 2);

  if (readWordFromCommand(command, 5, 10)) {
      _absolute_humidity = absolute_humidity;
      return true;
  } else {
    return false;
  }
}

bool Sgp30::readWordFromCommand(const uint8_t command[], const uint8_t commandLength, const uint16_t delayms, uint16_t *readdata, const uint8_t readlen)
{
    if (_callbacks->write(_i2caddr, command, commandLength) != MICROBIT_OK) {
        return false;
    }

    _callbacks->delay_ms_relaxed(delayms);

    if (readlen == 0) {
        return true;
        }

    uint8_t replylen = readlen * (SGP30_WORD_LEN +1);
    uint8_t replybuffer[replylen];
    if (_callbacks->read(_i2caddr,replybuffer,replylen) != MICROBIT_OK) {
        return false;
    }
    
    for (uint8_t i=0; i<readlen; i++) {
        uint8_t crc = generateCRC(replybuffer+i*3, 2);
        if (crc != replybuffer[i * 3 + 2])
        return false;
        // success! store it
        readdata[i] = replybuffer[i*3];
        readdata[i] <<= 8;
        readdata[i] |= replybuffer[i*3 + 1];
    }
    
    return true;
}

uint8_t Sgp30::generateCRC(const uint8_t *data, const uint8_t datalen) {
  // calculates 8-Bit checksum with given polynomial
  uint8_t crc = SGP30_CRC8_INIT;

  for (uint8_t i=0; i<datalen; i++) {
    crc ^= data[i];
    for (uint8_t b=0; b<8; b++) {
      if (crc & 0x80)
    crc = (crc << 1) ^ SGP30_CRC8_POLYNOMIAL;
      else
    crc <<= 1;
    }
  }
  return crc;
}