#include "buffered_serial.h"

BufferedSerial::BufferedSerial(PinName tx, PinName rx) : Serial(tx,rx), led1(LED1), led2(LED2), rx_sem(0), tx_sem(0)
{
    tx_in=0;
    tx_out=0;
    rx_in=0;
    rx_out=0;

    device_irqn = UART0_IRQn;

    // attach the interrupts
    Serial::attach(this, &BufferedSerial::Rx_interrupt, Serial::RxIrq);
    Serial::attach(this, &BufferedSerial::Tx_interrupt, Serial::TxIrq);
}

// Copy tx line buffer to large tx buffer for tx interrupt routine
void BufferedSerial::send_line(char *c)
{
    int i;
    char temp_char;
    bool empty;
    i = 0;
    strncpy(tx_line,c,LINE_SIZE);
    // Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(device_irqn);
    empty = (tx_in == tx_out);
    while ((i==0) || (tx_line[i-1] != '\n')) {
        // Wait if buffer full
        if (IS_TX_FULL) {
            // End Critical Section - need to let interrupt routine empty buffer by sending
            NVIC_EnableIRQ(device_irqn);
            //while (IS_TX_FULL) ; // buffer is full
            tx_sem.wait();
            // Start Critical Section - don't interrupt while changing global buffer variables
            NVIC_DisableIRQ(device_irqn);
        }
        tx_buffer[tx_in] = tx_line[i];
        i++;
        tx_in = NEXT(tx_in);
    }
    if (Serial::writeable() && (empty)) {
        temp_char = tx_buffer[tx_out];
        tx_out = NEXT(tx_out);
        // Send first character to start tx interrupts, if stopped
        LPC_UART0->THR = temp_char;
    }
    // End Critical Section
    NVIC_EnableIRQ(device_irqn);
}

// Read a line from the large rx buffer from rx interrupt routine
void BufferedSerial::read_line(char *c)
{
    int i;
    i = 0;
    // Start Critical Section - don't interrupt while changing global buffer variables
    NVIC_DisableIRQ(device_irqn);
    // Loop reading rx buffer characters until end of line character
    while ((i==0) || (rx_line[i-1] != '\n')) {
        // Wait if buffer empty
        if (IS_RX_EMPTY) { // buffer empty
            // End Critical Section - need to allow rx interrupt to get new characters for buffer
            NVIC_EnableIRQ(device_irqn);
            //while (rx_in == rx_out) ; // buffer empty
            rx_sem.wait();
            // Start Critical Section - don't interrupt while changing global buffer variables
            NVIC_DisableIRQ(device_irqn);
        } else {
            rx_sem.wait();
        }
        rx_line[i] = rx_buffer[rx_out];
        i++;
        rx_out = NEXT(rx_out);
        
        // prevent overflow on rx_line
        if(i == LINE_SIZE) 
        {
            i--;
            break;
        }
    }
    rx_line[i++] = 0;
    // End Critical Section
    NVIC_EnableIRQ(device_irqn);
    strncpy(c,rx_line,i);
}

// Interupt Routine to read in data from serial port
void BufferedSerial::Rx_interrupt()
{
    uint32_t IRR0= LPC_UART0->IIR;
    led1=1;
    while (readable() && !(IS_RX_FULL)) {
        rx_buffer[rx_in] = LPC_UART0->RBR;
        rx_in = NEXT(rx_in);
        rx_sem.release();
    }
    led1=0;
}

// Interupt Routine to write out data to serial port
void BufferedSerial::Tx_interrupt()
{
    uint32_t IRR = LPC_UART0->IIR;
    led2=1;
    while ((writeable()) && (tx_in != tx_out)) { // while serial is writeable and there are still characters in the buffer
        LPC_UART0->THR = tx_buffer[tx_out]; // send the character
        tx_out = NEXT(tx_out);
    }
    if(!IS_TX_FULL) // if not full
        tx_sem.release();
    led2=0;
}