#include "mbed.h"
#include "rtos.h"
#include "power.h"
#include "pinout.h"
#include "pll.h"
#include "jtag.h"
#include "mmap.h"
#include "GPIOInterrupt.h"
#include "basic_io.h"
#include "panic.h"
#include "main.h"
#include "signal.h"

// pointer to thread
Thread *thread_ptr;

void init_hw(void);
void req_intr_handler(void);
void print_rambuffer(void const *args);

int main()
{
    init_hw();
    
    JTAG jtag;
    // We check if JTAG works well by checking idcode returned by issuing a test instruction and read result idcode back.
    int idcode = jtag.readID();
    if(idcode != 0x4ba00477) {
        panic("ERROR: IDCode %X\r\n", idcode);
    }
    
    // create working threads first
    Thread core_handler(listen_req, (void *)(&jtag));
    Thread debugThread(print_rambuffer, (void *)(&jtag));
    thread_ptr = &core_handler;
    // enable GPIO interrupt
    enable_GPIO_intr(&req_intr_handler);

    jtag.reset();
    jtag.leaveState();
    jtag.PowerupDAP();
    // setup necessary internal clock source selection
    jtag.writeMemory(intclk_source, 2);
    jtag.writeMemory(extclk_source, 1);
    jtag.writeMemory(ext_div_by, 10);
    power_core(1);
    
    set_pll_frequency (200, jtag);
    
    // Begin to load program
    mbed_printf("Begining loading program.\r\n");
    if (jtag.loadProgram()) {
        mbed_printf("Load Failed!\r\n");
    } else {
        mbed_printf("Load Succeed!\r\n");
        CORERESETn = 0;
        CORERESETn = 1;        
        char line[80];
        
        while (1) {
            
            mbed_printf("Type 'quit' to quit.\r\n");
            mbed_scanf("%s", line);
            if (strncmp(line, "quit", 80) == 0)
                break;
            else if (strncmp(line, "debug", 80) == 0)
                debugThread.signal_set(SIG_DEBUG);
            
        }
        
    }    
    
    core_handler.terminate();
    thread_ptr = NULL;
    jtag.reset();
    
    mbed_printf("Powering Down\r\n");
    power_down();
    mbed_printf("Done.\r\n");
    while (1)
        ; 
}

void init_hw(void) {
    float core_volt = 1;
    power_down();
    power_up(core_volt); // Power Up Chip
    mbed_printf("Powered up!\r\n");

    PORESETn = 0;
    CORERESETn = 0;
    wait_us(100);
    PORESETn = 1;
    CORERESETn = 1;
}

void req_intr_handler(void) {
    if (thread_ptr)
        thread_ptr->signal_set(SIG_COREREQ);
    return;
}

void print_rambuffer(void const *args) {
    static bool init = true;
    JTAG *pJtag = (JTAG *)args;
    while (1) {
        Thread::signal_wait(SIG_DEBUG);
        if (init) {
            pJtag->reset();
            pJtag->leaveState();
            pJtag->PowerupDAP();
            init = false;
        }
        debug_print(pJtag);
    }
}
