/*
 * Implementations of two threads created in main.cpp.
 * One thread is used for handling core read/write request, the other thread is for sending an acknowledge interrupt.
 *
 * Written by Zimin Wang
 *  
 */
#include "mbed.h"
#include "rtos.h"
#include "jtag.h"
#include "mmap.h"
#include "GPIOInterrupt.h"
#include "basic_io.h"
#include "signal.h"

#define PRINT_REQ   (0U)
#define SCAN_REQ    (1U)

// interrupt pins
static InterruptIn req_intr(p15);
// p16 is used to send an IRQ1 to core. 
static DigitalOut ack_intr(p16);

void enable_GPIO_intr(void (*handler)(void)) {
    req_intr.rise(handler);
    req_intr.mode(PullDown);
}

void disable_GPIO_intr(void) {
    req_intr.disable_irq();
    return;  
}


// GPIO_out should be put in main loop. When an output request interrupt from chip occurs,
// this function read output data from chip and print data. Then it sends an output acknowledge
// signal to unblock task in the chip. 
// When there is no output request interrupt it returns quietly.
void listen_req(void const *args) {
    static bool init = true;
    
    JTAG *pJtag = (JTAG *)args;
    while (1) {
        Thread::signal_wait(SIG_COREREQ);
        if (init) {
            // inittialize jtag
            pJtag->reset();
            pJtag->leaveState();
            pJtag->PowerupDAP();
            init = false;
        }
        // check whether it's a print request or scan request
        unsigned int type = pJtag->readMemory(IO_TYPE);
        if (type == PRINT_REQ) {
            // read message from JTAG
            term_printf(pJtag);
        } else if (type == SCAN_REQ) {
            term_scanf(pJtag);
        } else {
            mbed_printf("Unknown request type: %u\r\n", type);
            continue;
        }
        // generate an acknowledge interrupt signal
        ack_intr = 1;
        ack_intr = 0;
    }
}


/*
// GPIO_in should be put in main loop. When an input request interrupt from chip occurs,
// this functino read user input and write user input to the chip. Then it sends an input
// acknowledge interrupt to unblock task in the chip.
// When there is no input request interrupt it returns quietly.
void GPIO_in(void const *args) {
    while (1) {
        mbed_printf("Enter GPIO in.\r\n");
        // generate interrupt signal
        in_ack = 1;
        in_ack = 0;
        Thread::wait(4000);
    }
}
*/