
#include "ADXL345.h"

ADXL345::ADXL345(PinName mosi, PinName miso, PinName sck, PinName cs) : _spi(mosi, miso, sck), _ncs (cs)
{
    _spi.frequency(2000000);
    _spi.format(8,3);
    _ncs = 1;
    wait_us(500);
    
    //oneByteWrite(ADXL345_SPI_WRITE | (ADXL345_POWER_CTL_REG & 0x3F), 0x00);
    oneByteWrite(ADXL345_POWER_CTL_REG, 0x08); 
    //Measure bit -> 1
    //A setting of 0 in the measure bit places the part into standby mode, and a setting of 1 places the part into measurement mode. 
    //The ADXL345 powers up in standby mode with minimum power consumption. 

    //oneByteWrite(ADXL345_SPI_WRITE | (ADXL345_DATA_FORMAT_REG & 0x3F), /*0x0B*/ 0x08);
    oneByteWrite(ADXL345_DATA_FORMAT_REG, /*0x0B*/ 0x00);
                                                                                            // |   D7      D6       D5     D4     D3       D2    [D1  D0] | 
                                                                                            // |SELF_TEST  SPI  INT_INVERT  0  FULL_RES  Justify  Range   |
                                                                                            // |    0       0        0      0      0        0      0   0  |
    //FULL_RES Bit -> 0
    //When this bit is set to a value of 1, the device is in full resolution 
    //mode, where the output resolution increases with the grange 
    //set by the range bits to maintain a 4 mg/LSB scale factor. When 
    //the FULL_RES bit is set to 0, the device is in 10-bit mode, and 
    //the range bits determine the maximum grange and scale factor. 

    //setDataRate(ADXL345_100HZ);
    setDataRate(ADXL345_200HZ);
    
    oneByteWrite(ADXL345_POWER_CTL_REG, 0x08);
    
    wait_us(500);
}

void ADXL345::setDataRate(int dataRate)
{
    //Get the current register contents, so we don't clobber the power bit.
    char registerContents = oneByteRead(ADXL345_BW_RATE_REG);

    registerContents &= 0x10;
    registerContents |= dataRate;

    oneByteWrite(ADXL345_BW_RATE_REG, registerContents);
}

void ADXL345::getAxes(int* axes)
{
    getData(axes, 3);
}

void ADXL345::getData(int* dataVector, int vectorSize)
{
    char* buffer = new char[vectorSize * 2];
    
    multiByteRead(ADXL345_DATAX0_REG, buffer, vectorSize * 2);
    
    for(int i = 0, j = 0; i < vectorSize; i++, j += 2)
    {
        dataVector[i] = (int16_t)((int)buffer[j+1] << 8 | (int)buffer[j]);
    }
    
    free(buffer);
}

float ADXL345::getX()
{
    char lsb_byte = 0;
    signed short msb_byte;
    float acc;
    
    lsb_byte = oneByteRead(ADXL345_DATAX1_REG);
    msb_byte = lsb_byte << 8;
    msb_byte |= oneByteRead(ADXL345_DATAX0_REG);
    acc = (float)msb_byte;
    return acc;
}

float ADXL345::getY()
{
    char lsb_byte = 0;
    signed short msb_byte;
    float acc;
    
    lsb_byte = oneByteRead(ADXL345_DATAY1_REG);
    msb_byte = lsb_byte << 8;
    msb_byte |= oneByteRead(ADXL345_DATAY0_REG);
    acc = (float)msb_byte;
    return acc;
}

float ADXL345::getZ()
{
    char lsb_byte = 0;
    signed short msb_byte;
    float acc;
    
    lsb_byte = oneByteRead(ADXL345_DATAZ1_REG);
    msb_byte = lsb_byte << 8;
    msb_byte |= oneByteRead(ADXL345_DATAZ0_REG);
    acc = (float)msb_byte;
    return acc;
}

int ADXL345::oneByteRead(int address) {
 
    int tx = (ADXL345_SPI_READ | (address & 0x3F));
    int rx = 0;
 
    _ncs = 0;
    //Send address to read from.
    _spi.write(tx);
    //Read back contents of address.
    rx = _spi.write(0x00);
    _ncs = 1;
 
    return rx;
 
}
 
void ADXL345::oneByteWrite(int address, char data) {
 
    int tx = (ADXL345_SPI_WRITE | (address & 0x3F));
 
    _ncs = 0;
    //Send address to write to.
    _spi.write(tx);
    //Send data to be written.
    _spi.write(data);
    _ncs = 1;
 
}
 
void ADXL345::multiByteRead(int startAddress, char* buffer, int size) {
 
    int tx = (ADXL345_SPI_READ | ADXL345_MULTI_BYTE | (startAddress & 0x3F));
 
    _ncs = 0;
    //Send address to start reading from.
    _spi.write(tx);
 
    for (int i = 0; i < size; i++) {
        buffer[i] = _spi.write(0x00);
    }
 
    _ncs = 1;
 
}
 
void ADXL345::multiByteWrite(int startAddress, char* buffer, int size) {
 
    int tx = (ADXL345_SPI_WRITE | ADXL345_MULTI_BYTE | (startAddress & 0x3F));
 
    _ncs = 0;
    //Send address to start reading from.
    _spi.write(tx);
 
    for (int i = 0; i < size; i++) {
        buffer[i] = _spi.write(0x00);
    }
 
    _ncs = 1;
 
}
