#include "SerialBuffered.h"

/**
 * Create a buffered serial class.
 *
 * @param tx A pin for transmit.
 * @param rx A pin for receive.
 */
SerialBuffered::SerialBuffered(PinName tx, PinName rx) : Serial(tx, rx)
{
    indexContentStart = 0;
    indexContentEnd = 0;
    timeout = 1;
    
    attach(this, &SerialBuffered::handleInterrupt);
}

/**
 * Destroy.
 */
SerialBuffered::~SerialBuffered()
{
}

/**
 * Set timeout for getc().
 *
 * @param ms milliseconds. (-1:Disable timeout)
 */
void SerialBuffered::setTimeout(int ms)
{
    timeout = ms;
}

/**
 * Read requested bytes.
 *
 * @param bytes A pointer to a buffer.
 * @param requested Length.
 *
 * @return Readed byte length.
 */
size_t SerialBuffered::readBytes(uint8_t *bytes, size_t requested)
{
    int i = 0;
    
    while (i < requested)
    {
        int c = getc();
        
        if (c < 0)
        {
            break;
        }
        
        bytes[i] = c;
        i++;
    }
    
    return i;
}

/**
 * Get a character.
 *
 * @return A character. (-1:timeout)
 */
int SerialBuffered::getc()
{
    timer.reset();
    timer.start();
    
    while (indexContentStart == indexContentEnd)
    {
        wait_ms(1);
        
        if ((timeout > 0) && (timer.read_ms() > timeout))
        {
            /*
             * Timeout occured.
             */
            // printf("Timeout occured.\n");
            return EOF;
        }
    }
    
    timer.stop();

    uint8_t result = buffer[indexContentStart++];
    indexContentStart =  indexContentStart % BUFFERSIZE;

    return result;
}

/**
 * Returns 1 if there is a character available to read, otherwise.
 */
int SerialBuffered::readable()
{
    return indexContentStart != indexContentEnd;
}

void SerialBuffered::handleInterrupt()
{
    while (Serial::readable())
    {
        if (indexContentStart == ((indexContentEnd + 1) % BUFFERSIZE))
        {
            /*
             * Buffer overrun occured.
             */
            // printf("Buffer overrun occured.\n");
            Serial::getc();
        }
        else
        {
            buffer[indexContentEnd++] = Serial::getc();
            indexContentEnd = indexContentEnd % BUFFERSIZE;
        }
    }
}

float SerialBuffered::f_readIntTo(char delimiter)
{
    char buffer[16];
    int i;
    float result;
    for (i = 0;; i++)
    {
        //while (!readable());
        
        char number = getc();
        
        if (number == delimiter)
            break;
            
        buffer[i] = number;     
    }
    
    buffer[i-1] = '\0';
    
    result = atof(buffer);
    
    return result;
}


int SerialBuffered::i_readIntTo(char delimiter)
{
    char buffer[16];
    int i;
    float result;
    for (i = 0;; i++)
    {
        //while (!readable());
        
        char number = getc();
        
        if (number == delimiter)
            break;
            
        buffer[i] = number;     
    }
    
    buffer[i-1] = '\0';
    
    result = atoi(buffer);
    
    return result;
}

void SerialBuffered::writeText(char* text)
{
    for (int i = 0; text[i] != '\0' && i < BUFFER_TEXT_SIZE; i++)
    {    
        while (!Serial::writeable());

        Serial::putc(text[i]);
    }
}
