
/* Libraries */

#include "mbed.h"
#include "mDot.h"
#include "ChannelPlans.h"
#include "RadioEvent.h"
#include "dot_util.h"

#if defined(FOTA)
    #include "Fota.h"
    #include "SPIFBlockDevice.h"
    #include "DataFlashBlockDevice.h"
#endif

/*****************************************************************************/

/* Connection Parameters */

// Channel Plan
// The active channel plan is the one that will be compiled
// CP_US915, CP_AU915, CP_EU868, CP_KR920, CP_AS923, CP_AS923_2, CP_AS923_3,
// CP_AS923_JAPAN, CP_AS923_JAPAN1, CP_AS923_JAPAN2, CP_IN865, CP_RU864
#if !defined(CHANNEL_PLAN)
    #define CHANNEL_PLAN CP_EU868
#endif

static uint8_t network_address[] = { 0x01, 0x02, 0x03, 0x04 };
static uint8_t network_session_key[] = { 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04 };
static uint8_t data_session_key[] = { 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04, 0x01, 0x02, 0x03, 0x04 };
static uint8_t frequency_sub_band = 6;
static lora::NetworkType network_type = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 5;
static uint8_t ack = 1;
static bool adr = true;

/*****************************************************************************/

/* Global Elements */

mDot* dot = NULL;
lora::ChannelPlan* plan = NULL;

/*****************************************************************************/

/* Main Function */

int main()
{
    RadioEvent events;

    // Create channel plan
    plan = create_channel_plan();
    assert(plan);

    #if defined(FOTA)
        mbed::BlockDevice* ext_bd = NULL;
    
        ext_bd = new SPIFBlockDevice();
        int ret = ext_bd->init();
        if (ext_bd->init() < 0)
        {
            delete ext_bd;
            ext_bd = new DataFlashBlockDevice();
            ret = ext_bd->init();
            // Check for zero size because DataFlashBlockDevice doesn't
            // return an error if the chip is not present
            if ((ret < 0) || (ext_bd->size() == 0))
            {
                delete ext_bd;
                ext_bd = NULL;
            }
        }
    
        dot = mDot::getInstance(plan, ext_bd);
    
        if (ext_bd != NULL)
        {
            logInfo("External flash device detected, type: %s, size: 0x%08x",
                    ext_bd->get_type(), (uint32_t)ext_bd->size());
        }
        
        Fota::getInstance(dot);
    #else
        dot = mDot::getInstance(plan);
    #endif
    assert(dot);

    // attach the custom events handler
    dot->setEvents(&events);

    if (!dot->getStandbyFlag()) {
        logInfo("mbed-os library version: %d.%d.%d", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        dot->resetConfig();
        dot->resetNetworkSession();

        // make sure library logging is turned on
        dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

        // update configuration if necessary
        if (dot->getJoinMode() != mDot::MANUAL) {
            logInfo("changing network join mode to MANUAL");
            if (dot->setJoinMode(mDot::MANUAL) != mDot::MDOT_OK) {
                logError("failed to set network join mode to MANUAL");
            }
        }
        // in MANUAL join mode there is no join request/response transaction
        // as long as the Dot is configured correctly and provisioned correctly on the gateway, it should be able to communicate
        // network address - 4 bytes (00000001 - FFFFFFFE)
        // network session key - 16 bytes
        // data session key - 16 bytes
        // to provision your Dot with a Conduit gateway, follow the following steps
        //   * ssh into the Conduit
        //   * provision the Dot using the lora-query application: http://www.multitech.net/developer/software/lora/lora-network-server/
        //      lora-query -a 01020304 A 0102030401020304 <your Dot's device ID> 01020304010203040102030401020304 01020304010203040102030401020304
        //   * if you change the network address, network session key, or data session key, make sure you update them on the gateway
        // to provision your Dot with a 3rd party gateway, see the gateway or network provider documentation
        update_manual_config(network_address, network_session_key, data_session_key, frequency_sub_band, network_type, ack);

        // enable or disable Adaptive Data Rate
        dot->setAdr(adr);

        // Configure the join delay
        dot->setJoinDelay(join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!dot->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        dot->restoreNetworkSession();
    }

    while(1)
    {
        
    }

    return 0;
}

/*****************************************************************************/
