/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@see "Joystick" by Craig Evans
@author Jordan Harper
@date   April 2015
*/

#include "mbed.h"
#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

#define DIRECTION_TOLERANCE 0.1 /*!< change this to alter tolerance of joystick direction */

#define USR_POWERDOWN    (0x104)

/**
@namespace button
@brief sends digital signal to mbed when joystick is pressed, can interrupt at any point in code where there is a button flag
*/
InterruptIn button(p17);

/**
@namespace xPot
@brief sends analog signal to mbed from X-axis of joystick
*/
AnalogIn xPot(p15);

/**
@namespace yPot
@brief sends analog signal to mbed from Y-axis of joystick
*/
AnalogIn yPot(p16);

/**
@namespace lcd
@brief recieves and sends data between LCD connections and mbed
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);

/**
@namespace power
@brief PWM Output to LED to signal that the power is On/Off
*/
PwmOut power(p24);

/**
@namespace buz
@brief PWM Output to Piezo Buzzer to play a song
*/
PwmOut buz(p21);


/* LCD dimentions */
int nx= 84; /*!< screen is 84 pixels across */
int ny= 48; /*!< by 48 pixels downwards */


int x = 0; /*!< wall starts from the coordinates (0,0) */
int y = 0; /*!< wall starts from the coordinates (0,0) */
int z = 30; /*!< gap in the wall starts near the middle of the screen */ 

/* initial coordinates for player object (i,j) */
int i=24; /*!< initial coordinates for player object (24,42) */
int j=42; /*!< initial coordinates for player object (24,42) */


int score = 0; /*!< initial value for the score at beginning of game */


int printFlag = 0; /*!< print flag initial value */


int FLAG = 0; /*!< initial value for flag that are used for wall movement */
int FLAG2 = 0; /*!< initial value for flag that are used for wall movement 2 */
int FLAG3 = 0; /*!< initial value for flag that are used for wall movement 3 */

int c = 0; /*!< initial value for position of option in menu*/

int bright = 0; /*!< initial case for brightness function in switch loop */

int vol = 0; /*!< initial case for volume function in switch loop */

int u = 0; /*!< initial value for u (used to select next frequency of PWM in song) */

int soundFlag = 1; /*!< flag for toggling mute */

int flashFlag = 0; /*!< flag for calling "Game Over!" phrase */

int BFlag = 0; /*!< flag for joystick button */

int sleepCounter = 0; /*!< initial value set for timer to save power */


int a = 0; /*!< wall starts from the coordinates (0,47) */
int b = 47; /*!< wall starts from the coordinates (0,47) */
int d = 30; /*!< gap in wall is near the middle of screen */
int f = 0; /*!< wall starts from the coordinates (0,0) */
int g = 0; /*!< wall starts from the coordinates (0,0) */
int h = 24; /*!< gap in wall is near the top left corner */
int q = 83; /*!< wall starts from the coordinates (83,0)*/
int w = 0; /*!< wall starts from the coordinates (83,0) */
int e = 24; /*!< gap in wall is near the top left corner */

float frequency[]= {880,880,0,0,440,880,0,0,440,880,600,1047,0,0,0,0,880,880,0,0,440,880,0,0,440,880,600,587,0,0,0,0,0}; /*!< frequency array for gameplay song */
float frequency2[]= {600,600,400,400,200}; /*!< frequency array for gameover song */


Ticker pollJoystick; /*!< timer to regularly read the joystick */
Ticker wallMovement; /*!<  timers for moving the walls */
Ticker wallMovement2; /*!< timers for moving the walls */
Ticker wallMovement3; /*!<  timers for moving the walls*/
Ticker flash; /*!< timer for "Game Over!" phrase and sleep intterupt */




Serial serial(USBTX,USBRX); /*!< Serial for debug */

LocalFileSystem local("local");  /*!< create local filesystem */

/* functions */

/** reads default positions of the joystick to calibrate later readings.
@param joystick.x0 - float value in X-axis to read
@param joystick.y0 - float value in Y-axis to read
@returns a definition for the centre value of joystick
*/
void calibrateJoystick();

/**  reads current joystick values relative to the calibrated centre value and calculates direction of joystick.
@param joystick.x - new float value in X-axis to read
@param joystick.y - new float value in Y-axis to read
@param joystick.button - state of joystick button to read
@param joystick.direction - calculated direction state of joystick
@returns direction of joystick
*/
void updateJoystick();

/** Turns all cells on the display off
@param i - all cells in x direction on screen
@param j - all cells in y direction on screen
@returns a blank screen
*/
void clearCells();


void fallingWall();

/** flag function for moving wall to call later */
void flagForWall();

/** flag function for moving wall to call later */
void flagForWall2();

/** flag function for moving wall to call later */
void flagForWall3();

/** displays menu of options that can be selected with joystick
@param case 0 - displays the "Play Game" option
@param case 1 - displays the "Settings" option
@param case 2 - displays the "Highscore" option
@returns a menu of options that can be selected to initiate other functions
*/
void menu();

/** initiates game and detects collisions with walls
@param i - x coordinate of player object
@param j - y coordinate of player object
@param m - value of specific case
@param case 0 - displays YES option
@param case 1 - displays restart question
@param case 2 - displays NO option
*/
void playGame();

/** uses UNIX time to set time and date of Real Time Clock 
*/
void setTime();

/** 
writes data to on board Flash memory.
 data such as the score and date (if it is better than the previous highscore) is printed to a text file.
*/
void writeDataToFile();

/** menu for brightness of LCD and volume of buzzer 
@param t - value of specific case
@param case 0 - brightness option
@param case 1 - volume option
@param case 2 - instructions option
*/
void BandVMenu();

/** sets brightness of LCD 
@param bright - value of specific case
@param case 0 - sets LCD backlight to 60%
@param case 1 - sets LCD backlight to 80%
@param case 2 - sets LCD backlight to 90%
@param case 3 - sets LCD backlight to 100%
@param case -1 - sets LCD backlight to 40%
@param case -2 - sets LCD backlight to 20%
@param case -3 - turns LCD backlight off
*/
void brightness();

/** sets volume of buzzer 
@param vol - value of specific case
@param case 0 - sets PWM duty ratio to buzzer to 0.6
@param case 1 - sets PWM duty ratio to buzzer to 0.8
@param case 2 - sets PWM duty ratio to buzzer to 0.9
@param case 3 - sets PWM duty ratio to buzzer to 1.0
@param case -1 - sets PWM duty ratio to buzzer to 0.4
@param case -2 - sets PWM duty ratio to buzzer to 0.2
@param case -3 - sets PWM duty ratio to buzzer to zero (MUTE)
*/
void volume();

/**
plays song using different frequencies of PWM through buzzer
used to play gameplay music
@param u - variable to represent each frequency in the array
*/
void tone();

/** 
plays song using different frequencies of PWM through buzzer
used to play a 'game over' tone
@param l - variable to represent each frequency in the array
*/
void deadTone();

/** reads data from the on board Flash memory. 
*/
void readDataFromFile();

/** checks current score against the highscore and decides whether to record it or not. 
this is done by scanning the text file saved to the on board Flash memory and then comparing it to the score that has been achieved after the current game is over.
*/
void scoreCheck();

/** flag function for when the game is over */
void flagForFlash();

/** displays "Game Over!" for 3 seconds
uses a timer instead of wait() function to save power
*/
void GameOverFlash();

/** Puts mbed into sleep mode and powers down the LCD if the game is left for more than 20 seconds.
this saves power. it can be turned back on again by pushing the joystick button.
*/
void powerSave();

/** puts mbed into sleep mode */
void Sleep();

/**
rules to prevent player object going off the screen
*/
void boundries();

/** moves player object on LCD screen to new position.
if joystick direction is a diagonal the player object moves one space in y direction and one space in x direction.
i.e. upRight direction would cause the player object to move up first, then to the right.
*/
void playerMovement();

/** sets different wall objects that move across the screen.
each wall has one gap in it which the player must move through to survive.
as the score increases, more walls are added to the screen making it more difficult to survive. 
*/
void fallingWalls();

/** 
gives instructions on how to navigate in menu
*/  
void instructions();

/**
flag function for joystick button
*/
void buttonFlag();

/* create enumerated type (0,1,2,3 etc. for direction) */
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    upLeft,
    downLeft,
    RIGHT,
    upRight,
    downRight,
    CENTRE,
    UNKNOWN
};



typedef struct JoyStick Joystick; /*!< struct for Joystick */

struct JoyStick {
    float x;    /*!< current x value */
    float x0;   /*!< 'centred' x value */
    float y;    /*!< current y value */
    float y0;   /*!< 'centred' y value */
    int button; /*!< button state (assume pull-down used, so 1 = pressed, 0 = unpressed) */
    DirectionName direction;  /*!< current direction */
};


Joystick joystick; /*!< create struct variable */


        
        
        