#include "mbed.h"
#include "HTTPServer.h"
#include "RequestHandler.h"
#include "EthernetInterface.h"
#include "MbedJSONValue.h"

#define SERVER_PORT 80
//#define DHCP
//#define DEBUG

//-- I2C OLED --
#include "Adafruit_SSD1306.h"

class I2CPreInit : public I2C
{
public:
    I2CPreInit(PinName sda, PinName scl) : I2C(sda, scl)
    {
        frequency(100000);
        start();
    };
};

//-- I2C OLED --
I2CPreInit gI2C(PA_10,PA_9);
Adafruit_SSD1306_I2c gOled(gI2C,NC,0x78,64,128);

//-- PWM DC --
PwmOut DC(D6);

//-- GPIO LED --
DigitalInOut GP05(D5);

DigitalInOut GP04(D4);
DigitalInOut GP03(D3);


//-- ADC --
AnalogIn   ain(A0);

EthernetInterface eth;
HTTPServer WIZwikiWebSvr;
MbedJSONValue WIZwikiREST;

// Enter a MAC address for your controller below.
uint8_t mac_addr[6] = {0x00, 0x08, 0xDC, 0x00, 0x01, 0xFE};
char mac_str[20];
char ip_addr[]      = "192.168.100.100";
char subnet_mask[]  = "255.255.255.0";
char gateway_addr[] = "192.168.100.1";
float ain_value;

GetRequestHandler  myGetReq;
//PostRequestHandler myPostReq;
PutRequestHandler  myPutReq;

//-- I2C OLED --
bool oled_set(void* param)
{
    char * oled;
    if(!param) return false;
    oled = (char*) param;
    gOled.clearDisplay();
    gOled.setTextCursor(0,0);
    gOled.printf("%s",oled);
    gOled.display();    
    return true;
}
//-- PWM DC --
bool pwm_set(void* param)
{
    if(!param) return false;
    DC.write((float)(*(int*)param)/100.0);
    return true;
}
//-- GPIO --
bool p5_set(void* param)
{
    if(!param) return false;
    GP05.write(*(int*)param);
    return true;
}
bool p4_set(void* param)
{
    if(!param) return false;
    GP04.write(*(int*)param);
    return true;
}
bool p3_set(void* param)
{
    if(!param) return false;
    GP03.write(*(int*)param);
    return true;
}

//-- ADC --
bool ain_read(void* param)
{
    ((MbedJSONValue*)param)->_value.asDouble = ain.read();
    return true;
}

void debug_info()
{
    printf("SP:0x%X\r\n",__current_sp());
    __heapstats((__heapprt)fprintf,stderr);
#ifdef DEBUG
    __heapvalid((__heapprt)fprintf,stderr, 1);
#endif
    printf("\r\n");
}    

void WIZwiki_REST_init();

int main(void)
{
#ifdef DEBUG
    debug_info();
#endif
    sprintf(mac_str, "%02X:%02X:%02X:%02X:%02X:%02X",mac_addr[0],mac_addr[1],mac_addr[2],mac_addr[3],mac_addr[4],mac_addr[5]);
    
    // OLED init
    gOled.begin();
    gOled.clearDisplay();
    
    // PWM init
    DC.period_ms(1);
    DC.write(0);
    
    //GPIO set & init
    GP05.output();
    GP05.write(1);
    
    //ADC init
    
    printf("START \r\n");    
    printf("sizeof(MbedJSONValue)=%d\r\n",sizeof(MbedJSONValue));
    printf("sizeof(vector)=%d\r\n",sizeof(std::vector<string*>));
    printf("sizeof(string)=%d\r\n",sizeof(std::string));
#ifdef DEBUG
    debug_info();
#endif

    WIZwiki_REST_init();

#ifdef DEBUG
    debug_info();
#endif
                    
    // Serialize it into a JSON string
    printf("---------------------WIZwikiREST-------------------- \r\n");
    printf("\r\n%s\r\n", WIZwikiREST.serialize().c_str());
    printf("---------------------------------------------------- \r\n");

    WIZwikiWebSvr.add_request_handler("GET", &myGetReq);
    //WIZwikiWebSvr.add_request_handler("POST", &myPostReq);
    WIZwikiWebSvr.add_request_handler("PUT", &myPutReq);
    //WIZwikiWebSvr.add_request_handler("DELETE", new PostRequestHandler());
    
    #ifdef DHCP
        eth.init(mac_addr); //Use DHCP
    #else
        eth.init(mac_addr, ip_addr, subnet_mask, gateway_addr); //Not Use DHCP
    #endif
    
    
    printf("Check Ethernet Link\r\n");
    
    do{
        printf("   Link - Wait... \r\n");
        wait(1);
    }while(!eth.ethernet_link());
    printf("-- Ethetnet PHY Link - Done -- \r\n");
        
    if (eth.connect() < 0 )
        printf("-- EThernet Connect - Fail -- \r\n");
    else
    {
        printf("-- Assigned Network Information -- \r\n");
        printf("   IP   : %s\r\n\r\n", eth.getIPAddress()); 
        printf("   MASK : %s\r\n\r\n", eth.getNetworkMask());
        printf("   GW   : %s\r\n\r\n", eth.getGateway());
    }
    
    printf("Link up\r\n");
    printf("IP Address is %s\r\n", eth.getIPAddress());
#ifdef DEBUG
    debug_info();
#endif

    if(!WIZwikiWebSvr.init(SERVER_PORT))
    {
        eth.disconnect();
        return -1;
    }

    while(1)
    {
        WIZwikiWebSvr.run();
    }
}

void WIZwiki_REST_init(void)
{
    //Fill the object
    WIZwikiREST["Name"] = "WIZwikiREST-io ver1.01";
    WIZwikiREST["Name"].accessible = false;
#ifdef DEBUG
    debug_info();
#endif
    
    //Network
    WIZwikiREST["Network"]["MAC"] = mac_str;
    WIZwikiREST["Network"]["IP"] = ip_addr; 
    WIZwikiREST["Network"]["IP"].accessible = true; 
    WIZwikiREST["Network"]["SN"] = subnet_mask;  
    WIZwikiREST["Network"]["SN"].accessible = true;  
    WIZwikiREST["Network"]["GW"] = gateway_addr;
    WIZwikiREST["Network"]["GW"].accessible = true;
    
    // I2C OLED
    WIZwikiREST["I2C"]["OLED"] = "none";
    WIZwikiREST["I2C"]["OLED"].accessible = true;
    WIZwikiREST["I2C"]["OLED"].cb_action = oled_set;
    
    // PWM DC
    WIZwikiREST["PWM"]["DC"] = DC.read();
    WIZwikiREST["PWM"]["DC"].accessible = true;
    WIZwikiREST["PWM"]["DC"].cb_action = pwm_set;
    
    // GPIO
    WIZwikiREST["GPIOs"]["P03"] = GP03.read();
    WIZwikiREST["GPIOs"]["P03"].accessible = true;
    WIZwikiREST["GPIOs"]["P03"].cb_action = p3_set;
    WIZwikiREST["GPIOs"]["P04"] = GP04.read();
    WIZwikiREST["GPIOs"]["P04"].accessible = true;
    WIZwikiREST["GPIOs"]["P04"].cb_action = p4_set;\
    WIZwikiREST["GPIOs"]["P05"] = GP05.read();
    WIZwikiREST["GPIOs"]["P05"].accessible = true;
    WIZwikiREST["GPIOs"]["P05"].cb_action = p5_set;
    
    // ADC
    WIZwikiREST["ADC"]["A0"] = ain.read();
    WIZwikiREST["ADC"]["A0"].accessible = false;
    WIZwikiREST["ADC"]["A0"].cb_action = ain_read;

#ifdef DEBUG
    debug_info();
#endif
}
