/* NeatGUI Library
 * Copyright (c) 2013 Neil Thiessen
 * Copyright (c) 2017 Jon Buckman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef SEPS525_SPI_H
#define SEPS525_SPI_H
 
#include "mbed.h"
#include "Display.h"
 
/** SEPS525_SPI class.
 *  Used for controlling an SEPS525-based OLED display connected to SPI.
 */
class SEPS525_SPI : public Display
{
public:
 
    /** Create an SEPS525 object connected to the specified SPI pins with the specified /CS and DC pins
     *
     * @param mosi The SPI data out pin.
     * @param miso The SPI data in pin.
     * @param sclk The SPI clock pin.
     * @param cs The SPI chip select pin.
     * @param dc The data/command pin.
     */
    SEPS525_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName dc, PinName res);
 
    /** Probe for the SEPS525 and initialize it if present
     *
     * @returns
     *   'true' if the device exists on the bus,
     *   'false' if the device doesn't exist on the bus.
     */
    virtual bool open();
 
    /** Send the buffer to the SEPS525
     */
    virtual void flush();
 
    /** Get the current state of the SEPS525
     *
     * @returns The current state as a Display::State enum.
     */
    virtual Display::State state();
 
    /** Set the state of the SEPS525
     *
     * @param mode The new state as a Display::State enum.
     */
    virtual void state(State s);
 
    /** Set the master current
    *
    * @param current The master current value.
    */
    virtual void masterCurrent(int current);

    /** Draw a single pixel at the specified coordinates
    *
    * @param x The X coordinate.
    * @param y The Y coordinate.
    * @param c The color of the pixel as a 32-bit ARGB value.
    */
    virtual void drawPixel(int x, int y, unsigned int c);
 
private:
    //Commands
    enum Command {
        CMD_INDEX                      = 0x00,
        CMD_STATUS_RD                  = 0x01,
        CMD_OSC_CTL                    = 0x02,
        CMD_IREF                       = 0x80,
        CMD_CLOCK_DIV                  = 0x03,
        CMD_REDUCE_CURRENT             = 0x04,
        CMD_SOFT_RST                   = 0x05,
        CMD_DISP_ON_OFF                = 0x06,
        CMD_PRECHARGE_TIME_R           = 0x08,
        CMD_PRECHARGE_TIME_G           = 0x09,
        CMD_PRECHARGE_TIME_B           = 0x0A,
        CMD_PRECHARGE_CURRENT_R        = 0x0B,
        CMD_PRECHARGE_CURRENT_G        = 0x0C,
        CMD_PRECHARGE_CURRENT_B        = 0x0D,
        CMD_DRIVING_CURRENT_R          = 0x10,
        CMD_DRIVING_CURRENT_G          = 0x11,
        CMD_DRIVING_CURRENT_B          = 0x12,
        CMD_DISPLAY_MODE_SET           = 0x13,
        CMD_RGB_IF                     = 0x14,
        CMD_RGB_POL                    = 0x15,
        CMD_MEMORY_WRITE_MODE          = 0x16,
        CMD_MX1_ADDR                   = 0x17,
        CMD_MX2_ADDR                   = 0x18,
        CMD_MY1_ADDR                   = 0x19,
        CMD_MY2_ADDR                   = 0x1A,
        CMD_MEMORY_ACCESS_POINTER_X    = 0x20,
        CMD_MEMORY_ACCESS_POINTER_Y    = 0x21,
        CMD_DDRAM_DATA_ACCESS_PORT     = 0x22,
        CMD_GRAY_SCALE_TABLE_INDEX     = 0x50,
        CMD_GRAY_SCALE_TABLE_DATA      = 0x51,
        CMD_DUTY                       = 0x28,
        CMD_DSL                        = 0x29,
        CMD_D1_DDRAM_FAC               = 0x2E,
        CMD_D1_DDRAM_FAR               = 0x2F,
        CMD_D2_DDRAM_SAC               = 0x31,
        CMD_D2_DDRAM_SAR               = 0x32,
        CMD_SCR1_FX1                   = 0x33,
        CMD_SCR1_FX2                   = 0x34,
        CMD_SCR1_FY1                   = 0x35,
        CMD_SCR1_FY2                   = 0x36,
        CMD_SCR2_SX1                   = 0x37,
        CMD_SCR2_SX2                   = 0x38,
        CMD_SCR2_SY1                   = 0x39,
        CMD_SCR2_SY2                   = 0x3A,
        CMD_SCREEN_SAVER_CONTEROL      = 0x3B,
        CMD_SS_SLEEP_TIMER             = 0x3C,
        CMD_SCREEN_SAVER_MODE          = 0x3D,
        CMD_SS_SCR1_FU                 = 0x3E,
        CMD_SS_SCR1_MXY                = 0x3F,
        CMD_SS_SCR2_FU                 = 0x40,
        CMD_SS_SCR2_MXY                = 0x41,
        CMD_MOVING_DIRECTION           = 0x42,
        CMD_SS_SCR2_SX1                = 0x47,
        CMD_SS_SCR2_SX2                = 0x48,
        CMD_SS_SCR2_SY1                = 0x49,
        CMD_SS_SCR2_SY2                = 0x4A,
    };
 
    //SPI interface variables
    SPI m_SPI;
    DigitalOut m_CS;
    DigitalOut m_DC;
    DigitalOut m_RES;
 
    //Command and data helpers
    void writeCommand(char command);
    void writeData(char data);
};
 
#endif
