/* NeatGUI Library
 * Copyright (c) 2013 Neil Thiessen
 * Copyright (c) 2017 Jon Buckman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include "SEPS525_SPI.h"

SEPS525_SPI::SEPS525_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName dc, PinName res) : Display(160, 128),  m_SPI(mosi, miso, sclk), m_CS(cs), m_DC(dc), m_RES(res)
{
    //Deselect the display
    m_CS = 1;
 
    //Set the SPI format to 8 bit data, high steady state clock, second edge capture
    m_SPI.format(8, 3);
 
    //Set the SPI frequency to 20MHz
    m_SPI.frequency(10000000);
}

bool SEPS525_SPI::open()
{
    //Initialize interface and reset display driver chip
    m_CS = 1;
    wait(0.01);
    m_RES = 0;
    wait(0.001);
    m_RES = 1;
    wait(0.01);

    //Init sequence for 160x128 OLED module
    writeCommand(CMD_DISP_ON_OFF);
    writeData(0x00);    //Display off

    writeCommand(CMD_OSC_CTL);
    writeData(0x01); //Export1 internal clock and OSC operates with external resistor

    writeCommand(CMD_REDUCE_CURRENT);
    writeData(0x00);

    writeCommand(CMD_CLOCK_DIV);
    writeData(0x30); //Clock div ratio 1: freq setting 90Hz

    
    writeCommand(CMD_IREF);
    writeData(0x00); //Iref controlled by external resistor

    writeCommand(CMD_PRECHARGE_TIME_R);
    writeData(0x01);
    writeCommand(CMD_PRECHARGE_TIME_G);
    writeData(0x01);
    writeCommand(CMD_PRECHARGE_TIME_B);
    writeData(0x01);

    writeCommand(CMD_PRECHARGE_CURRENT_R);
    writeData(0x0A);
    writeCommand(CMD_PRECHARGE_CURRENT_G);
    writeData(0x0A);
    writeCommand(CMD_PRECHARGE_CURRENT_B);
    writeData(0x0A);

    writeCommand(CMD_DRIVING_CURRENT_R);
    writeData(0x52); //R = 82uA
    writeCommand(CMD_DRIVING_CURRENT_G);
    writeData(0x38); //R = 56uA
    writeCommand(CMD_DRIVING_CURRENT_B);
    writeData(0x3A); //R = 58uA

    writeCommand(CMD_DISPLAY_MODE_SET);
    writeData(0x00); //RGB, column=0-159, column data display=Normal display

    
    writeCommand(CMD_RGB_IF);
    writeData(0x01); //External interface mode=MPU

    //6 bits triple transfer, 262K support, Horizontal address counter is increased,
    //vertical address counter is increased. The data is continuously written
    //horizontally
    writeCommand(CMD_MEMORY_WRITE_MODE);
    writeData(0x76);

    writeCommand(CMD_MX1_ADDR);  //Column start
    writeData(0x00);
    writeCommand(CMD_MX2_ADDR);  //Column end
    writeData(width()-1);
    writeCommand(CMD_MY1_ADDR);  //row start
    writeData(0x00);
    writeCommand(CMD_MY2_ADDR);  //row end
    writeData(height()-1);

    writeCommand(CMD_MEMORY_ACCESS_POINTER_X);  //X
    writeData(0x00);
    writeCommand(CMD_MEMORY_ACCESS_POINTER_Y);  //Y
    writeData(0x00);

    writeCommand(CMD_DUTY);
    writeData(0x7F);

    writeCommand(CMD_DSL);
    writeData(0x00);

    writeCommand(CMD_D1_DDRAM_FAC);  //X
    writeData(0x00);
    writeCommand(CMD_D1_DDRAM_FAR);  //Y
    writeData(0x00);

    writeCommand(CMD_SCR1_FX1);  //Screen saver columns start
    writeData(0x00);
    writeCommand(CMD_SCR1_FX2);  //Screen saver columns end
    writeData(width()-1);
    writeCommand(CMD_SCR1_FY1);  //screen saver row start
    writeData(0x00);
    writeCommand(CMD_SCR1_FY2);  //Screen saver row end
    writeData(height()-1);

    writeCommand(CMD_DISP_ON_OFF);
    writeData(0x01); //Display on

    //Return success
    return true;
}

void SEPS525_SPI::flush()
{
}

Display::State SEPS525_SPI::state()
{
    //Return the base class's state
    return Display::state();
}

void SEPS525_SPI::state(State s)
{
    //Check what the requested state is
    if (s == Display::DISPLAY_ON) {
        //Turn the display on
        writeCommand(CMD_DISP_ON_OFF);
        writeData(0x01); //Display on
    } else if (s == Display::DISPLAY_OFF) {
        //Turn the display off
        writeCommand(CMD_DISP_ON_OFF);
        writeData(0x00);    //Display off
    }
 
    //Update the base class
    Display::state(s);
}

void SEPS525_SPI::masterCurrent(int current)
{
    if(current < 8)
    {
        writeCommand(CMD_REDUCE_CURRENT);
        writeData(0x04);
    }
    else
    {
        writeCommand(CMD_REDUCE_CURRENT);
        writeData(0x00);
    }
}

void SEPS525_SPI::drawPixel(int x, int y, unsigned int c)
{
    //Range check the pixel
    if ((x < 0) || (x >= width()) || (y < 0) || (y >= height()))
        return;
 
    writeCommand(CMD_MEMORY_ACCESS_POINTER_X);  // memory accesspointer x
    writeData(x);
    writeCommand(CMD_MEMORY_ACCESS_POINTER_Y);  // memory accesspointer y
    writeData(y);

    writeCommand(CMD_DDRAM_DATA_ACCESS_PORT);
    writeData((c & 0xFF0000) >> 16);
    writeData((c & 0x00FF00) >> 8);
    writeData(c & 0xFF);
}

void SEPS525_SPI::writeCommand(char command)
{
    //Set DC to command and select the display
    m_DC = 0;
    m_CS = 0;
 
    //Write the command byte
    m_SPI.write(command);
 
    //Deselect the display
    m_CS = 1;
}

void SEPS525_SPI::writeData(char data)
{
    //Set DC to data and select the display
    m_DC = 1;
    m_CS = 0;
 
    //Write the data byte
    m_SPI.write(data);
 
    //Deselect the display
    m_CS = 1;
}
