/* NeatGUI Library
 * Copyright (c) 2013 Neil Thiessen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#ifndef SSD1353_SPI_H
#define SSD1353_SPI_H
 
#include "mbed.h"
#include "Display.h"
 
/** SSD1353_SPI class.
 *  Used for controlling an SSD1353-based OLED display connected to SPI.
 */
class SSD1353_SPI : public Display
{
public:
 
    /** Create an SSD1353 object connected to the specified SPI pins with the specified /CS and DC pins
     *
     * @param mosi The SPI data out pin.
     * @param miso The SPI data in pin.
     * @param sclk The SPI clock pin.
     * @param cs The SPI chip select pin.
     * @param dc The data/command pin.
     */
    SSD1353_SPI(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName dc);
 
    /** Probe for the SSD1353 and initialize it if present
     *
     * @returns
     *   'true' if the device exists on the bus,
     *   'false' if the device doesn't exist on the bus.
     */
    virtual bool open();
 
    /** Send the buffer to the SSD1353
     */
    virtual void flush();
 
    /** Get the current state of the SSD1353
     *
     * @returns The current state as a Display::State enum.
     */
    virtual Display::State state();
 
    /** Set the state of the SSD1353
     *
     * @param mode The new state as a Display::State enum.
     */
    virtual void state(State s);
 
    /** Set the master current
    *
    * @param current The master current value.
    */
    virtual void masterCurrent(int current);

    /** Draw a single pixel at the specified coordinates
    *
    * @param x The X coordinate.
    * @param y The Y coordinate.
    * @param c The color of the pixel as a 32-bit ARGB value.
    */
    virtual void drawPixel(int x, int y, unsigned int c);
 
private:
    //Commands
    enum Command {
        CMD_SETCOLUMN       = 0x15,
        CMD_SETROW          = 0x75,
        CMD_WRITERAM        = 0x5C,
        CMD_READRAM         = 0x5D,
        CMD_CONTRASTA       = 0x81,
        CMD_CONTRASTB       = 0x82,
        CMD_CONTRASTC       = 0x83,
        CMD_MASTERCURRENT   = 0x87,
        CMD_PRECHARGESPEED2 = 0x8A,
        CMD_SETREMAP        = 0xA0,
        CMD_STARTLINE       = 0xA1,
        CMD_DISPLAYOFFSET   = 0xA2,
        CMD_NORMALDISPLAY   = 0xA4,
        CMD_DISPLAYALLON    = 0xA5,
        CMD_DISPLAYALLOFF   = 0xA6,
        CMD_INVERTDISPLAY   = 0xA7,
        CMD_MUXRATIO        = 0xA8,
        CMD_DIMMODESET      = 0xAB,
        CMD_DISPLAYDIM      = 0xAC,
        CMD_DISPLAYOFF      = 0xAE,
        CMD_DISPLAYON       = 0xAF,
        CMD_PHASE12PERIOD   = 0xB1,
        CMD_CLOCKDIV        = 0xB3,
        CMD_RECHARGEPERIOD2 = 0xB4,
        CMD_SETGRAY         = 0xB8,
        CMD_USELUT          = 0xB9,
        CMD_PRECHARGELEVEL  = 0xBB,
        CMD_VCOMH           = 0xBE,
        CMD_SOFTRESET       = 0xE2,
        CMD_NOP             = 0xE3,
        CMD_COMMANDLOCK     = 0xFD,
        CMD_DRAWLINE        = 0x21,
        CMD_DRAWRECTANGLE   = 0x22,
        CMD_COPY            = 0x23,
        CMD_DIMWINDOW       = 0x24,
        CMD_CLEARWINDOW     = 0x25,
        CMD_FILLENABLE      = 0x26,
        CMD_CONTHVSCROLL    = 0x27,
        CMD_HORIZSCROLLOFF  = 0x2E,
        CMD_HORIZSCROLLON   = 0x2F,
        CMD_VERTSCROLLAREA  = 0xA3
    };
 
    //SPI interface variables
    SPI m_SPI;
    DigitalOut m_CS;
    DigitalOut m_DC;
 
    //Caching variables
    int m_StartX;
    int m_StartY;
    int m_CursX;
    int m_CursY;
    char m_Cache[1024];
    int m_CacheIndex;
 
    //Command and data helpers
    void writeCommand(char command);
    void writeData(char data);
};
 
#endif
