/* NeatGUI Library
 * Copyright (c) 2013 Neil Thiessen
 * Copyright (c) 2017 Jon Buckman
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef ILI9163_H
#define ILI9163_H

#include "mbed.h"
#include "Display.h"

/** ILI9163 class.
 *  Used for controlling an ILI9163-based TFT display.
 */
class ILI9163 : public Display
{
public:
    /** Create an ILI9163 object connected to the specified SPI pins with the specified /CS and DC pins
     *
     * @param mosi The SPI data out pin.
     * @param miso The SPI data in pin.
     * @param sclk The SPI clock pin.
     * @param sclk The SPI chip select pin.
     * @param sclk The data/command pin.
     */
    ILI9163(PinName D0, PinName D1, PinName D2, PinName D3, PinName D4, PinName D5, PinName D6, PinName D7, PinName cs, PinName dc, PinName wr, PinName rst);

    /** Probe for the ILI9163 and initialize it if present
     *
     * @returns
     *   'true' if the device exists on the bus,
     *   'false' if the device doesn't exist on the bus.
     */
    virtual bool open();

    /** Send the buffer to the ILI9163
     */
    virtual void flush();

    /** Get the current state of the ILI9163
     *
     * @returns The current state as a Display::State enum.
     */
    virtual Display::State state();

    /** Set the state of the ILI9163
     *
     * @param mode The new state as a Display::State enum.
     */
    virtual void state(State s);

    //void display();

    /** Draw a single pixel at the specified coordinates
    *
    * @param x The X coordinate.
    * @param y The Y coordinate.
    * @param c The color of the pixel as a 32-bit ARGB value.
    */
    virtual void drawPixel(int x, int y, unsigned int c);

private:
    //Interface variables
    BusOut  bus;
    DigitalOut m_CS;
    DigitalOut m_DC;
    DigitalOut m_WR;
    DigitalOut m_RST;

    //Command and data helpers
    void writeCommand(char command);
    void writeData8(char data);
    void writeData16(unsigned short data);
};

#endif
