/**
 * @file ldc1000.h
 *
 * @author Jon Buckman
 * 
 * @section LICENSE
 *
 * Copyright (c) 2014 Jon Buckman
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * @section DESCRIPTION
 *
 * LDC1000 Inductance to Digital Converter from Texas Instruments.
 *
 * Datasheet:
 *
 * http://www.ti.com/lit/ds/symlink/ldc1000.pdf
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ldc1000.h"
 *
 * Serial pc(USBTX, USBRX); // tx, rx
 *
 * LDC1000 ldc(D11, D12, D13, D10, D9);   // mosi, miso, sck, cs, irq
 * 
 * int main() {
 *    while(1) {
 *       pc.printf("The device revision ID is %d\n", ldc.getDeviceID());
 *       wait(1);
 *    }
 * }
 * @endcode
 */

#ifndef __LDC1000_H__
#define __LDC1000_H__

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define LDC1000_MIN_SENSOR_FREQUENCY 27
#define LDC1000_MAX_SENSOR_FREQUENCY 234
#define LDC1000_AMP_1V 0x00
#define LDC1000_AMP_2V 0x08
#define LDC1000_AMP_4V 0x10
#define LDC1000_RESP_192 0x02
#define LDC1000_RESP_384 0x03
#define LDC1000_RESP_768 0x04
#define LDC1000_RESP_1536 0x05
#define LDC1000_RESP_3072 0x06
#define LDC1000_RESP_6144 0x07
#define LDC1000_CLK_SEL 0x02
#define LDC1000_CLK_PD 0x01
#define LDC1000_INTB_MODE_DRDYB_ENBL 0x04
#define LDC1000_INTB_MODE_CMP_STAT 0x03
#define LDC1000_INTB_MODE_WAKE_ENBL 0x01
#define LDC1000_INTB_MODE_DSBL 0x00
#define LDC1000_PWR_MODE_ACT 0x01
#define LDC1000_PWR_MODE_STDBY 0x00
#define LDC1000_STAT_OSC 0x80
#define LDC1000_STAT_DRDY 0x40
#define LDC1000_STAT_WAKE 0x20
#define LDC1000_STAT_CMP 0x10

/**
 * LDC1000 Inductance to Digital Converter from Texas Instruments.
 */
class LDC1000 {

public:

    /**
     * Constructor.
     *
     * @param mosi mbed pin to use for MOSI line of SPI interface.
     * @param miso mbed pin to use for MISO line of SPI interface.
     * @param sck mbed pin to use for SCK line of SPI interface.
     * @param csn mbed pin to use for not chip select line of SPI interface.
     * @param irq mbed pin to use for the interrupt request line.
     */
    LDC1000(PinName mosi, PinName miso, PinName sck, PinName csn, PinName irq = NC);

    /**
     * Get the device ID.
     *
     * @return The device ID (8).
     */
    int getDeviceID(void);

    /**
     * Set the Rp maximum.
     *
     * @param rp Rp maximum (0x00 - 0x1f).
     */
    void setRpMaximum(int rp);

    /**
     * Get the Rp maximum.
     *
     * @return The device ID (0x00 - 0x1f).
     */
    int getRpMaximum(void);

    /**
     * Set the Rp minimum.
     *
     * @param rp Rp maximum (0x20 - 0x3f).
     */
    void setRpMinimum(int rp);

    /**
     * Get the Rp minimum.
     *
     * @return The device ID (0x20 - 0x3f).
     */
    int getRpMinimum(void);

    /**
     * Set the sensor frequency.
     *
     * @param frequency sensor frequency (N = 68.94 * log10(F/2000)).
     */
    void setSensorFrequency(int frequency);

    /**
     * Get the sensor frequency.
     *
     * @return The sensor frequency (0 - 255).
     */
    int getSensorFrequency(void);

    /**
     * Set the oscillator amplitude.
     *
     * @param oscamp oscillator amplitude (0x00 - 1V, 0x01 - 2V, 0x10 - 4V).
     */
    void setOscAmplitude(int oscamp);

    /**
     * Get the oscillator amplitude.
     *
     * @return The oscillator amplitude (0x00 - 1V, 0x01 - 2V, 0x02 - 4V).
     */
    int getOscAmplitude(void);

    /**
     * Set the response time.
     *
     * @param resp response time (0x02 - 192, 0x03 - 384, 0x04 - 768, 0x05 - 1536, 0x06 - 3072, 0x07 - 6144).
     */
    void setResponseTime(int resp);

    /**
     * Get the response time.
     *
     * @return The response time (0x02 - 192, 0x03 - 384, 0x04 - 768, 0x05 - 1536, 0x06 - 3072, 0x07 - 6144).
     */
    int getResponseTime(void);

    /**
     * Set the clock select.
     *
     * @param sel clock select (0 - external time base, 1 - external crystal).
     */
    void setClockSelect(int sel);

    /**
     * Get the clock select.
     *
     * @return The clock select (0 - external time base, 1 - external crystal).
     */
    int getClockSelect(void);

    /**
     * Set the disable external time base clock.
     *
     * @param pd clock select (0 - enable external time base clock, 1 - disable external time base clock).
     */
    void setClockTimeBase(int pd);

    /**
     * Get the disable external time base clock.
     *
     * @return The clock select (0 - enable external time base clock, 1 - disable external time base clock).
     */
    int getClockTimeBase(void);

    /**
     * Set the least significant byte of comparator threshold high register.
     *
     * @param data data (0 - 255).
     */
    void setComparatorThresholdHighLSB(int data);

    /**
     * Get the least significant byte of comparator threshold high register.
     *
     * @return The register (0 - 255).
     */
    int getComparatorThresholdHighLSB(void);

    /**
     * Set the most significant byte of comparator threshold high register.
     *
     * @param data data (0 - 255).
     */
    void setComparatorThresholdHighMSB(int data);

    /**
     * Get the most significant byte of comparator threshold high register.
     *
     * @return The register (0 - 255).
     */
    int getComparatorThresholdHighMSB(void);

    /**
     * Set the least significant byte of comparator threshold low register.
     *
     * @param data data (0 - 255).
     */
    void setComparatorThresholdLowLSB(int data);

    /**
     * Get the least significant byte of comparator threshold low register.
     *
     * @return The register (0 - 255).
     */
    int getComparatorThresholdLowLSB(void);

    /**
     * Set the mode of the INTB pin.
     *
     * @param mode mode (0x00 - all modes disabled, 0x01 - wake up enable on INTB pin, 0x02 - INTB pin indicates status of comparator, 0x04 - DRDYB enable on INTB pin ).
     */
    void setINTBPin(int mode);

    /**
     * Get the mode of the INTB pin.
     *
     * @return The mode register (0x00 - all modes disabled, 0x01 - wake up enable on INTB pin, 0x02 - INTB pin indicates status of comparator, 0x04 - DRDYB enable on INTB pin ).
     */
    int getINTBPin(void);

    /**
     * Set the power mode configuraton.
     *
     * @param data power mode (0 - standby, 1 - active).
     */
    void setPower(int data);

    /**
     * Get the power mode configuration.
     *
     * @return The power mode (0 - standby, 1 - active).
     */
    int getPower(void);

    /**
     * Get the status.
     *
     * @return The status (0x10 - comparator, 0x20 - wake up, 0x40 - data ready, 0x80 - OSC status).
     */
    int getStatus(void);

    /**
     * Get the least significant byte of promiity data.
     *
     * @return The proximity data (0 - 255).
     */
    int getProximityDataLSB(void);

    /**
     * Get the most significant byte of promiity data.
     *
     * @return The proximity data (0 - 255).
     */
    int getProximityDataMSB(void);

    /**
     * Get the least significant byte of frequency counter data.
     *
     * @return The frequency counter data (0 - 255).
     */
    int getFrequencyCounterLSB(void);

    /**
     * Get the mid significant byte of frequency counter data.
     *
     * @return The frequency counter data (0 - 255).
     */
    int getFrequencyCounterMIDSB(void);

    /**
     * Get the most significant byte of frequency counter data.
     *
     * @return The frequency counter data (0 - 255).
     */
    int getFrequencyCounterMSB(void);

private:

    /**
     * Set the contents of an addressable register.
     *
     * @param regAddress address of the register
     * @param regData data to write to the register
     */
    void setRegister(int regAddress, int regData);

    /**
     * Get the contents of an addressable register.
     *
     * @param regAddress address of the register
     * @return The contents of the register
     */
    int getRegister(int regAddress);

    SPI         spi_;
    DigitalOut  nCS_;
    InterruptIn nIRQ_;

    int mode;

};

#endif /* __LDC1000_H__ */