/**
 * @file AD5668.h
 *
 * @author Jon Buckman
 * 
 * @section LICENSE
 *
 * Copyright (c) 2017 Jon Buckman
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * @section DESCRIPTION
 *
 * AD5668 Digital to Analog Converter from Analog Devices.
 *
 * Datasheet:
 *
 * http://www.analog.com/media/en/technical-documentation/data-sheets/AD5668.pdf
 *
 * Tie LDAC and CLR pins to VDD if not being used.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "AD5668.h"
 *
 * Serial pc(USBTX, USBRX); // tx, rx
 *
 * AD5668 dac(D11, D12, D13, D10);   // mosi, miso, sck, cs
 * 
 * int main() {
 *      while (1) {
 *          for (int i = 0; i < 65536; i+=10) {
 *              dac.AD5668_WRITE_INPUT_N_UPDATE_N(AD5668_ADDR_DAC_A, i);
 *              pc.printf("%d,", i);
 *              wait(.01);
 *          }
 *      }
 * }
 * @endcode
 */

#ifndef __AD5668_H__
#define __AD5668_H__

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
/* AD5668 Input Register */
#define AD5668_CMD(x)              ((0x0000000F & (unsigned int)(x)) << 24)
#define AD5668_ADDR(x)             ((0x0000000F & (unsigned int)(x)) << 20)
#define AD5628_DATA_BITS(x)        ((0x000FFFFF & (unsigned int)(x)) << 8)
#define AD5648_DATA_BITS(x)        ((0x000FFFFF & (unsigned int)(x)) << 6)
#define AD5668_DATA_BITS(x)        ((0x000FFFFF & (unsigned int)(x)) << 4)

/* Command Definitions (AD5668_COMMAND(x) options) */
#define AD5668_CMD_WRITE_INPUT_N             0 // Write to Input Register n.
#define AD5668_CMD_UPDATE_DAC_N              1 // Update DAC Register n.
#define AD5668_CMD_WRITE_INPUT_N_UPDATE_ALL  2 // Write to Input Register n, update all.
#define AD5668_CMD_WRITE_INPUT_N_UPDATE_N    3 // Write to and update DAC Channel n.
#define AD5668_CMD_POWERDOWN                 4 // Power down/power up DAC.
#define AD5668_CMD_LOAD_CLEAR_CODE           5 // Load clear code register.
#define AD5668_CMD_LOAD_LDAC_REG             6 // Load LDAC register.
#define AD5668_CMD_RESET                     7 // Reset (power-on reset)
#define AD5668_CMD_SET_INT_REF               8 // Set up internal REF register.

/* Address Commands (AD5668_ADDRESS(x) options) */
#define AD5668_ADDR_DAC_A      0x00 //DAC A
#define AD5668_ADDR_DAC_B      0x01 //DAC B
#define AD5668_ADDR_DAC_C      0x02 //DAC C
#define AD5668_ADDR_DAC_D      0x03 //DAC D
#define AD5668_ADDR_DAC_E      0x04 //DAC E
#define AD5668_ADDR_DAC_F      0x05 //DAC F
#define AD5668_ADDR_DAC_G      0x06 //DAC G
#define AD5668_ADDR_DAC_H      0x07 //DAC H
#define AD5668_ADDR_DAC_ALL    0x0F //All DACs

/* Internal Reference Register */
#define AD5668_INT_REF_OFF     0
#define AD5668_INT_REF_ON      1

/* Power-Down Modes of Operation */
#define AD5668_POWER_MODE(x)      ((0x03 & (unsigned short) (x)) << 8)

#define AD5668_PWRDN_NONE         0 // Normal operation
#define AD5668_PWRDN_1K           1 // 1 KOhm to GND    (Power-down mode)
#define AD5668_PWRDN_100K         2 // 100 KOhm to GND  (Power-down mode)
#define AD5668_PWRDN_3STATE       3 // Three-state      (Power-down mode)

/* Clear Code Function */
#define AD5668_CODE_0X0000        0
#define AD5668_CODE_0X8000        1
#define AD5668_CODE_0XFFFF        2
#define AD5668_CODE_NOP           3

/* SPI */
#define AD5668_SPI_MAX_DATA_RATE 50000000

/**
 * AD5668 Digital to Analog Converter from Analog Devices.
 */
class AD5668 {

public:

    /**
     * Constructor.
     *
     * @param mosi mbed pin to use for MOSI line of SPI interface.
     * @param miso mbed pin to use for MISO line of SPI interface.
     * @param sck mbed pin to use for SCK line of SPI interface.
     * @param csn mbed pin to use for not chip select line of SPI interface.
     */
    AD5668(PinName mosi, PinName miso, PinName sck, PinName csn);

    /**
     * Set power mode for each channnel.
     *
     * @param pwrMode power mode of the device.
     * @param channel the channel or channels that are being configured.
     */
    void AD5668_PowerMode(unsigned char pwrMode, unsigned char channel);

    /**
     * Reset the device.
     *
     * @param none.
     */
    void AD5668_Reset(void);

    /**
     * Write a data word to a channel.
     *
     * @param channel the channel or channels that are being configured.
     * @param data the DAC data word.
     */
    void AD5668_WriteInput(uint8_t channel, uint16_t data);

    /**
     * Latch a data word to a channel.
     *
     * @param channel the channel or channels that are being configured.
     * @param data the DAC data word.
     */
    void AD5668_UpdateDAC(uint8_t channel, uint16_t data);

    /**
     * Write a data word to a channel and latch all channels.
     *
     * @param channel the channel or channels that are being configured.
     * @param data the DAC data word.
     */
    void AD5668_WriteInputUpdateAll(uint8_t channel, uint16_t data);

    /**
     * Write a data word to a channel and latch the same channel.
     *
     * @param channel the channel or channels that are being configured.
     * @param data the DAC data word.
     */
    void AD5668_WriteInputUpdate(uint8_t channel, uint16_t data);

    /**
     * Set the internal voltage reference on or off.
     *
     * @param val set internal voltage reference.
     */
    void AD5668_InternalReference(uint8_t val);

    /**
     * Set the internal voltage reference on or off.
     *
     * @param val clear mode of the device.
     */
    void AD5668_ClearCode(uint8_t val);

private:

    /**
     * Writes a 32-bit data-word to the Input Register of the device.
     *
     * @param registerValue value of the register.
     */
    void AD5668_SetInputRegister(unsigned int registerValue);

    /**
     * Write array values to the SPI.
     *
     * @param reg pointer to the array
     */
    void SPI_Write(unsigned char *reg);

    SPI         spi_;
    DigitalOut  nCS_;

};

#endif /* __AD5668_H__ */