/**
 * @file ad5668.cpp
 *
 * @author Jon Buckman
 * 
 * @section LICENSE
 *
 * Copyright (c) 2017 Jon Buckman
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * @section DESCRIPTION
 *
 * AD5668 Digital to Analog Converter from Analog Devices.
 *
 * Datasheet:
 *
 * http://www.analog.com/media/en/technical-documentation/data-sheets/AD5668.pdf
 */
 
/**
 * Includes
 */
#include "ad5668.h"

/**
 * Defines
 */


/**
 * Methods
 */
AD5668::AD5668(PinName mosi,
                    PinName miso,
                    PinName sck, 
                    PinName csn) : spi_(mosi, miso, sck), nCS_(csn) {

    nCS_ = 1;

    spi_.frequency(AD5668_SPI_MAX_DATA_RATE/5);     // 2Mbit, 1/5th the maximum transfer rate for the SPI bus
    spi_.format(8,1);                                   // 8-bit, ClockPhase = 0, ClockPolarity = 1

    /* Behaves as a power-on reset. */
    AD5668_Reset();
    /* Turns on the on-board reference. */
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_SET_INT_REF)| AD5668_INT_REF_ON);
    /* Clear code is set to 0x0000. */
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_LOAD_CLEAR_CODE)| AD5668_CODE_0X0000);
}

void AD5668::AD5668_PowerMode(unsigned char pwrMode, unsigned char channel)
{
    unsigned char selectedChannel = 0;

    if(channel == AD5668_ADDR_DAC_ALL)
    {
        selectedChannel = 0xFF;
    }
    else
    {
        selectedChannel = (1 << channel);
    }
    /* Selects a power mode for the selected channel. */
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_POWERDOWN) |
                            AD5668_POWER_MODE(pwrMode) |
                            selectedChannel);
}

void AD5668::AD5668_Reset(void)
{
     AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_RESET));
}

void AD5668::AD5668_WriteInput(uint8_t channel, uint16_t data)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_WRITE_INPUT_N) | AD5668_ADDR(channel) | AD5668_DATA_BITS(data));
}

void AD5668::AD5668_UpdateDAC(uint8_t channel, uint16_t data)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_UPDATE_DAC_N) | AD5668_ADDR(channel) | AD5668_DATA_BITS(data));
}

void AD5668::AD5668_WriteInputUpdateAll(uint8_t channel, uint16_t data)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_WRITE_INPUT_N_UPDATE_ALL) | AD5668_ADDR(channel) | AD5668_DATA_BITS(data));
}

void AD5668::AD5668_WriteInputUpdate(uint8_t channel, uint16_t data)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_WRITE_INPUT_N_UPDATE_N) | AD5668_ADDR(channel) | AD5668_DATA_BITS(data));
}

void AD5668::AD5668_InternalReference(uint8_t val)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_SET_INT_REF) | val);
}

void AD5668::AD5668_ClearCode(uint8_t val)
{
    AD5668_SetInputRegister(AD5668_CMD(AD5668_CMD_LOAD_CLEAR_CODE) | val);
}

void AD5668::AD5668_SetInputRegister(unsigned int registerValue)
{
    unsigned char registerWord[4] = {0, 0, 0, 0};
    registerWord[0] = (unsigned char)((registerValue & 0xFF000000) >> 24);
    registerWord[1] = (unsigned char)((registerValue & 0x00FF0000) >> 16);
    registerWord[2] = (unsigned char)((registerValue & 0x0000FF00) >> 8);
    registerWord[3] = (unsigned char)((registerValue & 0x000000FF) >> 0);
    SPI_Write(registerWord);
}

void AD5668::SPI_Write(unsigned char *reg)
{
    nCS_=0;
    for (int i = 0; i < 4; i++) {
        spi_.write(reg[i]);
    }
    nCS_=1;
}