/* Sampling example for My Digital Signal Controller library
 * Copyright (c) 2019, Gastón H. SALAZAR-SILVA
 * SPDX-License-Identifier: Apache-2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 */

/*
 *  This program is an example of how to sample an analog input.
 *
 *  # Interface #
 *
 *  The connections are for a Nucleo f446re board. PC_1 and PA_4 should be physically connected.
 *      PC_1 (A4) <---> PA_4 (A2)
 */

#include "mbed.h"
#include "mydsc_ring_buffer.h"

AnalogOut       analog_out_1(PA_4); ///< Generated signal by the own board. Assigned to the pin PA_4.
AnalogIn        analog_in(PC_1);    ///< Input to be sampled, assigned to the pin PC_1. PC_1 and PA_4 should be physically connected.
AnalogOut       analog_out_2(PA_5); ///< Analog output 2, connected to the pin PA_5, which is also connected to USER_LED.

const double    PI      =   3.14159265358979323846;
const uint32_t  SIZE    =   1024;                   ///< Buffer size - 1
const float     H       =   1.0/(SIZE);          ///< Sampling period
const int32_t   PHI     =   -256;                   ///< Phase shift

mydsc_ring_buffer_t rb;         ///< circular queue or ring buffer.
Ticker              ticker;                             //Variable ticker

void ticker_isr() {
    static unsigned int    k    =   0;                  //Dato no accesado desde otra funcion

    analog_out_1 = ( 1.0 + sin(2*PI*k*H)) / 2.0;        //Genera su salida analogica. Salida senoidal
    mydsc_ring_buffer_push(&rb, analog_in);             //Apuntador RB e inserta el balor en PHI
    analog_out_2 = *mydsc_ring_buffer_sample(&rb, PHI); //Emular un arreglo, lee el valor con un apuntador
    k++;
}   

int main() {
    // Setup
    Serial tty(USBTX, USBRX);                           //Clase para el puerto Serial
    
    mydsc_ring_buffer_init(&rb, SIZE);                  //Se inicializa y se da el tamaño de memoria
    ticker.attach(&ticker_isr, H);
    tty.baud(115200);                                   //Velocidad a 115200 Baudios para el plotter
    
    while(1) { 
        // Loop
        tty.printf("%f\t%f\n", (*mydsc_ring_buffer_sample(&rb, 0)),(*mydsc_ring_buffer_sample(&rb, PHI)));      //
    }
}

