#include "AD7792.h"

#define SPI_DUMMY_BYTE   0x00

#define CS_HIGH     cs=1
#define CS_LOW      cs=0

DigitalOut cs(D10);
SPI spi(SPI_MOSI, SPI_MISO, SPI_SCK);
extern Serial pc;

static u8 SPIWriteByte(uint8_t byte)
{
    return spi.write(byte);
}

static u8 SPIReadByte(void)
{
    return spi.write(SPI_DUMMY_BYTE);
}

// Basic SPI routines to simplify code
// read and write one register
u8 SPIxreadOneRegister(u8 regAddress)
{
    u8 regValue = 0;

    CS_LOW;
    SPIWriteByte(AD7792_COMM_READ | AD7792_COMM_ADDR(regAddress));  // read instruction
    regValue = SPIReadByte();
    CS_HIGH;

    return regValue;
}

u16 SPIxreadTwoRegisters(u8 regAddress)
{
    u16 twoRegValue = 0;

    CS_LOW;
    SPIWriteByte(AD7792_COMM_READ | AD7792_COMM_ADDR(regAddress));  // read instruction
    twoRegValue = SPIReadByte();
    twoRegValue = (twoRegValue << 8) + SPIReadByte();
    CS_HIGH;

    return twoRegValue;
}

void SPIxwriteOneRegister(u8 regAddress, u8 regValue)
{
    CS_LOW;
    SPIWriteByte(AD7792_COMM_WRITE | AD7792_COMM_ADDR(regAddress));  // write instruction
    SPIWriteByte(regValue);
    CS_HIGH;
}

void SPIxwriteTwoRegisters(u8 regAddress, u16 twoRegValue)
{

    u8 twoRegValueH = twoRegValue >> 8;
    u8 twoRegValueL = twoRegValue;

    CS_LOW;
    SPIWriteByte(AD7792_COMM_WRITE | AD7792_COMM_ADDR(regAddress));  // write instruction
    SPIWriteByte(twoRegValueH);
    SPIWriteByte(twoRegValueL);
    CS_HIGH;
}

/***************************************************************************//**
 * @brief Sends 32 consecutive 1's on SPI in order to reset the part.
 *
 * @return  None.
 * When a reset is initiated, the user must allow a period of 500 μs before accessing any of the on-chip registers.
 *******************************************************************************/
void AD7792_Reset(void)
{
    CS_LOW;
    SPIWriteByte(0xFF);
    SPIWriteByte(0xFF);
    SPIWriteByte(0xFF);
    SPIWriteByte(0xFF);
    SPIWriteByte(0xFF);
    CS_HIGH;

}

/***************************************************************************//**
 * @brief Reads status reg.
 *
 * @param None.
 *
 * @return status.
 *******************************************************************************/
unsigned char AD7792_Status(void)
{
    u8 status = 0;
    status = SPIxreadOneRegister( AD7792_REG_STAT);

    return (status);
}

bool AD7792_WaitRdy()
{

    while((AD7792_STAT_RDY & AD7792_Status()))
    {}

    return true ;

}


/***************************************************************************//**
 * @brief Sets the operating mode of AD7792.
 *
 * @param mode - Mode of operation.
 *      AD7792_MODE_CONT
 *      AD7792_MODE_SINGLE
 *      AD7792_MODE_IDLE
 *      AD7792_MODE_PWRDN
 *      AD7792_MODE_CAL_INT_ZERO
 *      AD7792_MODE_CAL_INT_FULL
 *      AD7792_MODE_CAL_SYS_ZERO
 *      AD7792_MODE_CAL_SYS_FULL
 * @return  None.    
 *******************************************************************************/
u16 AD7792_SetMode(unsigned long mode)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_MODE);
    reg &= ~AD7792_MODE_SEL(0xFFFF);
    reg |= AD7792_MODE_SEL(mode);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);

//  reg=SPIxreadTwoRegisters(AD7792_REG_MODE);
    return reg;

}

u16 AD7792_POWER_DOWN(void)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_MODE);
    reg &= ~AD7792_MODE_SEL(0xFFFF);
    reg |= AD7792_MODE_SEL(AD7792_MODE_PWRDN);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);

//  reg=SPIxreadTwoRegisters(AD7792_REG_MODE);
//  debug_msg("AD7792_REG_MODE=%X\r\n", reg);

    return reg;

}

/***************************************************************************//**
 * @brief Sets the operating mode of AD7792.
 *
 * @param mode - Mode of operation.
 *      AD7792_MODE_RATE_470...AD7792_MODE_RATE_4_17
 *
 * @return  None.
 *******************************************************************************/
u16 AD7792_SetUpdateRate(u8 rate)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_MODE);
    reg &= ~AD7792_MODE_RATE(0xFFFF);
    reg |= AD7792_MODE_RATE(rate);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);

//  reg=SPIxreadTwoRegisters(AD7792_REG_MODE);
    return reg;
}

/***************************************************************************//**
 * @brief 设置内部电流源.
 *
 * @param mode - Mode of operation.
 *
 * @return  None.
 *******************************************************************************/
void AD7792_SetIEXC(u8 dir, u8 en)
{
    u16 reg;

    reg = AD7792_IEXCDIR(dir) | AD7792_IEXCEN(en);
    SPIxwriteOneRegister(AD7792_REG_IO, reg);
}

/***************************************************************************//**
 * @brief Selects the channel of AD7792.
 *
 * @param  channel - ADC channel selection.
 *
 * @return  None.    
 *******************************************************************************/
u16 AD7792_SetChannel(unsigned long channel)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_CONF); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_CONF_CHAN(0xFFFF);
    reg |= AD7792_CONF_CHAN(channel);
    SPIxwriteTwoRegisters(AD7792_REG_CONF, reg); // CS is modified by SPI read/write functions.

//  reg=SPIxreadTwoRegisters(AD7792_REG_CONF);
    return reg;

}

/***************************************************************************//**
 * @brief  Sets the bias.
 *
 * @param
 *  0 0 Bias voltage generator disabled
 *  0 1 Bias voltage connected to AIN1(.)
 *  1 0 Bias voltage connected to AIN2(.)
 *  1 1 Reserved
 *
 * @return  None.
 *******************************************************************************/
u16 AD7792_SetBias(u8 bias)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_CONF);
    reg &= ~AD7792_CONF_VBIAS(0xFFFF);
    reg |= AD7792_CONF_VBIAS(bias);
    SPIxwriteTwoRegisters(AD7792_REG_CONF, reg);

    reg=SPIxreadTwoRegisters(AD7792_REG_CONF);
    return reg;
}

/***************************************************************************//**
 * @brief  Sets the gain of the In-Amp.
 *
 * @param  gain - Gain.
 *
 *  For higher gains, the buffer is automatically enabled.
 *
 * @return  None.    
 *******************************************************************************/
u16 AD7792_SetGain(unsigned long gain)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_CONF); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_CONF_GAIN(0xFFFF);
    reg |= AD7792_CONF_GAIN(gain);
    SPIxwriteTwoRegisters(AD7792_REG_CONF, reg); // CS is modified by SPI read/write functions.

//  reg=SPIxreadTwoRegisters(AD7792_REG_CONF);
    return reg;
}

/***************************************************************************//**
 * @brief  Sets buffer mode.
 * 0, 1
 * @return  None.
 *******************************************************************************/
u16 AD7792_SetBuf(u8 buf)
{
    u16 reg;

    reg = SPIxreadTwoRegisters(AD7792_REG_CONF); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_CONF_BUF;
    if (buf)
        reg |= AD7792_CONF_BUF;
    SPIxwriteTwoRegisters(AD7792_REG_CONF, reg); // CS is modified by SPI read/write functions.

//  reg=SPIxreadTwoRegisters(AD7792_REG_CONF);
    return reg;
}

/***************************************************************************//**
 * @brief Sets the reference source for the ADC.
 *
 * @param type - Type of the reference.
 *               Example: AD7792_REFSEL_EXT - External Reference Selected
 *                        AD7792_REFSEL_INT - Internal Reference Selected.
 *
 * @return None.    
 *******************************************************************************/
u16 AD7792_SetIntReference(unsigned char type)
{
    u16 reg = 0;

    reg = SPIxreadTwoRegisters(AD7792_REG_CONF); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_CONF_REFSEL(0xFFFF);
    reg |= AD7792_CONF_REFSEL(type);
    SPIxwriteTwoRegisters(AD7792_REG_CONF, reg); // CS is modified by SPI read/write functions.

//  reg=SPIxreadTwoRegisters(AD7792_REG_CONF);
    return reg;
}

/***************************************************************************//**
 * @brief Performs the given calibration to the specified channel.
 *
 * @param mode - Calibration type.
 * @param channel - Channel to be calibrated.
 *
 * @return none.
 *******************************************************************************/
u16 AD7792_Calibrate(unsigned char mode, unsigned char channel)
{
    u16 reg;

    reg=AD7792_SetChannel(channel);
    reg &= SPIxreadTwoRegisters(AD7792_REG_MODE);
    reg &= ~AD7792_MODE_SEL(0x7);
    reg |= AD7792_MODE_SEL(mode);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);
    AD7792_WaitRdy();
    
    return  reg;
}

/***************************************************************************//**
 * @brief Returns the result of a single conversion.
 *
 * @return regData - Result of a single analog-to-digital conversion.
 *******************************************************************************/
unsigned short AD7792_SingleConversion(void)
{
    u16 reg = 0x0;

    reg = SPIxreadTwoRegisters(AD7792_REG_MODE); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_MODE_SEL(0xFFFF);
    reg |= AD7792_MODE_SEL(AD7792_MODE_SINGLE);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);

    return reg;
}

/***************************************************************************//**
 * @brief Returns the result of a single conversion.
 *
 * @return regData - Result of a single analog-to-digital conversion.
 *******************************************************************************/
unsigned short AD7792_ContinusConversion(void)
{
    u16 reg = 0x0;

    reg = SPIxreadTwoRegisters(AD7792_REG_MODE); // CS is modified by SPI read/write functions.
    reg &= ~AD7792_MODE_SEL(0xFFFF);
    reg |= AD7792_MODE_SEL(AD7792_MODE_CONT);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg);

    return reg;
}

/***************************************************************************//**
 * @brief Returns the average of several conversion results.
 *
 * @return samplesAverage - The average of the conversion results.
 *******************************************************************************/
unsigned long AD7792_ContinuousReadAvg(unsigned char sampleNumber)
{
    unsigned long samplesAverage = 0x0;
    unsigned long reg = 0x0;
    unsigned char count = 0x0;

    reg = AD7792_MODE_SEL(AD7792_MODE_CONT);
    SPIxwriteTwoRegisters(AD7792_REG_MODE, reg); // CS is not modified by SPI read/write functions.
    for (count = 0; count < sampleNumber; count++) {
        AD7792_WaitRdy();
        samplesAverage += SPIxreadTwoRegisters(AD7792_REG_DATA); // CS is not modified by SPI read/write functions.
    }
    samplesAverage = samplesAverage / sampleNumber;

    return (samplesAverage);
}

/*
 * TEMPERATURE SENSOR
 * Accuracy ±2 °C typ
 * Sensitivity 0.81 mV/°C typ
 * Applies if user calibrates the temperature sensor
 *
 * Automatically selects gain = 1 and internal reference
 */
float AD7792_Temperature_Sensor()
{
    u16 ad;
    float voltage;

    AD7792_SetChannel(AD7792_CH_TEMP);
    SPIxreadTwoRegisters( AD7792_REG_DATA);
    AD7792_SingleConversion();
    AD7792_WaitRdy();
    ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
    voltage = 1.17 * (ad - 32767) / 32767 * 6;
    return voltage;
}

/*
 * When Bit CH2 to Bit CH0 equal 1, the voltage on the AVDD pin is internally
 * attenuated by 6, and the resultant voltage is applied to the Σ-Δ modulator
 * using an internal 1.17 V reference for analog-todigital conversion.
 */
float AD7792_AVdd_Monitor()
{
    u16 ad;
    float voltage;

    AD7792_SetChannel(AD7792_CH_AVDD_MONITOR);
    AD7792_SingleConversion();
    AD7792_WaitRdy();
    ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
    voltage = 1.17 * (ad - 32767) / 32767 * 6;
    //debug_msg("V: %d\t%.3f\r\n", ad, voltage);
    return voltage;
}

void AD7792_checkAllRegs()
{

    pc.printf("AD7792   Read of all Control Regs:\r\n");

    pc.printf("\tSTATUS\t%02X\r\n", SPIxreadOneRegister(AD7792_REG_STAT));
    pc.printf("\tMODE\t%04X\r\n", SPIxreadTwoRegisters(AD7792_REG_MODE));
    pc.printf("\tCONF\t%04X\r\n", SPIxreadTwoRegisters(AD7792_REG_CONF));
    pc.printf("\tIO\t%02X\r\n", SPIxreadOneRegister(AD7792_REG_IO));

}

bool AD7792_Init()
{
    u16 reg;

    spi.format(8,3);
    spi.frequency(2000000);

    AD7792_Reset();

    reg=AD7792_SetBuf(1);
    reg=AD7792_SetIntReference(AD7792_REFSEL_INT);

    reg=SPIxreadOneRegister(AD7792_REG_ID);
    pc.printf("%x\n",reg);

    if (AD7792_ID == (0x0F & reg)) {
        return true;
    } else {
        return false;
    }
}

void AD7792_test(void)
{
    u16 reg, ad, ad1, ad2, t;
    u32 i = 0, n = 1500;
    float v;
    extern Serial pc;

    AD7792_SetUpdateRate(AD7792_MODE_RATE_242);

    pc.printf("\r\nNO.\tVDD\tTEMP\tCH1\tCH2\r\n", i);
    for (i = 0; i < n; i++) {
        pc.printf("%d\t", i);

        AD7792_SetChannel(AD7792_CH_AVDD_MONITOR);
        AD7792_SingleConversion();
        AD7792_WaitRdy();
        //delay(t);
        ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
        v = 1.17 * (ad - 32767) / 32767 * 6;
        pc.printf("%d(%.2f)\t", ad,v);

        AD7792_SetChannel(AD7792_CH_TEMP);
        AD7792_SingleConversion();
        AD7792_WaitRdy();
        //delay(t);
        ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
        v = 1.17 * (ad - 32767) / 32767 * 6;
        pc.printf("%d\t", ad);

        reg = AD7792_SetChannel(AD7792_CH_AIN1P_AIN1M);
        reg = AD7792_SetBias(1);
        reg = AD7792_SetGain(AD7792_GAIN_1);
        AD7792_SingleConversion();
        AD7792_WaitRdy();
        //delay(t);
        ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
        pc.printf("%d\t", ad);

        reg = AD7792_SetChannel(AD7792_CH_AIN2P_AIN2M);
        reg = AD7792_SetBias(2);
        reg = AD7792_SetGain(AD7792_GAIN_1);
        AD7792_SingleConversion();
        AD7792_WaitRdy();
        //delay(t);
        ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
        pc.printf("%d\t", ad);

        pc.printf("\r\n");
    }

//        //通道1：声波
//    AD7792_SetUpdateRate(AD7792_MODE_RATE_470);
//    AD7792_SetChannel(AD7792_CH_AIN1P_AIN1M);
//    AD7792_SetBias (AD7792_BIAS_AIN1);
//    AD7792_SetGain(AD7792_GAIN_1);
//    AD7792_ContinusConversion();
//    AD7792_checkAllRegs();
//    for (i = 0; i < n; i++) {
//        ad = SPIxreadTwoRegisters( AD7792_REG_DATA);
//        v = 1.17 * (ad - 32767) / 32767;
//        wait_ms(4);
//        buf[i]=ad;
//    }
//    for (i = 0; i < n; i++) {
//        pc.printf("%d\t%d\r\n", i,buf[i]);
//    }

}


