//
// Accellerometer project
//
// Laurence Halfpenny
// August 2015
 
#include <stdint.h>
#include <string.h>
#include "mbed.h"           // Needed for MBED library functions
#include "main.h"           // Needed for definitions of things that can be changed
#include "USBSerial.h"      // Needed for USB serial port
#include "DipCortex-EEprom.h"
#include "LPC13Uxx.h"          

// Define the X, Y & Z analogue inputs on ADCs
AnalogIn Z_in(P0_11);       // Z accelleration voltage
AnalogIn Y_in(P0_12);       // Y accelleration voltage
AnalogIn X_in(P0_13);       // X accelleration voltage

// On-board LED
DigitalOut LED(P0_1);

// Push button
// Logic FALSE when pushed, logic TRUE when no pushed
DigitalIn BUTTON(P0_16);

// Acceleration values
int Z_accelleration=0;
int Y_accelleration=0;
int X_accelleration=0;

// Storage arrays for X,Y,Z
// Pack them to save RAM space
#pragma pack(push,1)
uint16_t Z_store[MAXSAMPLES]={};
uint16_t Y_store[MAXSAMPLES]={};
uint16_t X_store[MAXSAMPLES]={};
#pragma pack(pop)

int Sample_index=0;
int Last_sample=0;
int EEPROM_index=0;

// Control variables
bool monitoring=FALSE;      // Controls whether monitoring is running or stopped

// Ticker for monitoring
Ticker monitor_tick;


// ------------------------------------------------------------------------------------------------------------
//  Read and Write the EEPROM 
// ------------------------------------------------------------------------------------------------------------


// ------------------------------------------------------------------------------------------------------------
//  Initialise the LPC1347 processor
// ------------------------------------------------------------------------------------------------------------
void init() 
{
    NVIC_SetPriority(SSP1_IRQn, 0x0); 
    NVIC_SetPriority(PIN_INT0_IRQn, 0x1);
    
    // SysTick set to lower priority than Wi-Fi SPI bus interrupt
    NVIC_SetPriority(SysTick_IRQn, 0x2); 
    
    
    // Enable RAM1
    LPC_SYSCON->SYSAHBCLKCTRL |= (0x1 << 26);
    
    // Clear the LED
    LED=FALSE;
    
    // Pull the BUTTON pin up to logic high
    BUTTON.mode(PullUp);
    
    // Initialise the IAP for EEPROM reading and writing
    IAP_Init();
}

// ------------------------------------------------------------------------------------------------------------
//  The monitoring software, triggered by the monitoring ticker
// ------------------------------------------------------------------------------------------------------------
void monitor_accelleration(void)
{
    LED = !LED;                         // Toggle the LED each sample
    
    X_accelleration=X_in.read_u16();    // Read and print X accelleration
    X_store[Sample_index]=X_accelleration;
    X_accelleration=(X_accelleration-(0xFFFF/2));  // Convert to change in accelleration 
    X_accelleration=X_accelleration/16;
    //pc.printf("X: %d ",X_accelleration);  

    Y_accelleration=Y_in.read_u16();    // Read and print Y accelleration
    Y_store[Sample_index]=Y_accelleration;
    Y_accelleration=(Y_accelleration-(0xFFFF/2));  // Convert to change in accelleration 
    Y_accelleration=Y_accelleration/16;
    //pc.printf("Y: %d ",Y_accelleration);  

    Z_accelleration=Z_in.read_u16();    // Read and print Z accelleration
    Z_store[Sample_index]=Z_accelleration;
    Z_accelleration=(Z_accelleration-(0xFFFF/2));  // Convert to change in accelleration 
    Z_accelleration=Z_accelleration/16;
    //pc.printf("Z: %d \r\n",Z_accelleration);  
    
    //pc.printf("Sample index: %d \r\n",Sample_index);  
    Sample_index++;
    
    if (Sample_index>=MAXSAMPLES)
        {
        //pc.printf("Stopping - full!\r\n");
        monitoring=FALSE;
        monitor_tick.detach();
        Last_sample=Sample_index;
        }
}

// ------------------------------------------------------------------------------------------------------------
//  main loop
// ------------------------------------------------------------------------------------------------------------
int main( void ) 
{   
int i;  // For controlling loops
char charIn = 0;
char prevCharIn = 0;
char Command = 0;
char j=0;    // For writing to clear EEPROM 

    // Initalise the LPC1347 & WiFiDIPCORTEX board
    init(); 
    
    // Wait for program to start
    wait(1);
    
    // If the BUTTON is pushed (logic FALSE) start logging otherwise go into keyboard menu
    if (BUTTON==FALSE)
        {
        
        // Start the monitoring
        Sample_index=0;
        monitoring=TRUE;
        monitor_tick.attach(&monitor_accelleration,MONITORINTERVAL);       
        
        // Wait for the monitoring to finish and then save it to EEPROM
        while (monitoring==TRUE)
            {
            // Just wait for monitoring to finish    
            wait(1);
            }
        
        // Now copy the storage arrays to EEPROM
        LED=TRUE;   // Turn LED solid on when writing to EEPROM
        IAP_Eeprom_Write(EEPROMBASEADDRESS, (uint8_t *)&X_store, sizeof(X_store));                                  // Write the X array
        IAP_Eeprom_Write(EEPROMBASEADDRESS+sizeof(X_store), (uint8_t *)&Y_store, sizeof(Y_store));                  // Write the Y array
        IAP_Eeprom_Write(EEPROMBASEADDRESS+sizeof(X_store)+sizeof(Y_store), (uint8_t *)&Z_store, sizeof(Z_store));  // Write the Z array
        
        // Now wait for ever just flashing LED
        while (TRUE)
            {
            LED=!LED;   // Flash the LED slowly at end
            wait(1);    
            }
        }
    else
        {
        USBSerial pc;               // open USB CDC serial port
        // Wait for terminal program to start
        wait(WAITATSTART);
        
        // Show the start banner
        pc.printf("\r\n");
        pc.printf("+-------------------------------------------+\r\n");
        pc.printf("|   3-axis accelleration measurer           |\r\n");
        pc.printf("|   Laurence Halfpenny                      |\r\n");
        pc.printf("|   Version 1.2                             |\r\n");
        pc.printf("+-------------------------------------------+\r\n");
    
        // Forever check for commands from the keyboard
        while (TRUE)
            {
            pc.printf("+-------------------------------------------+\r\n");
            pc.printf("|   V 1.2 MENU:                             |\r\n");
            pc.printf("|   1: Start monitoring                     |\r\n");
            pc.printf("|   2: Stop  monitoring                     |\r\n");
            pc.printf("|   3: Stop / Dump values                   |\r\n");
            pc.printf("|   4: Dump EEPROM values                   |\r\n");
            pc.printf("|   9: Clear EEPROM values                  |\r\n");
            pc.printf("+-------------------------------------------+\r\n");
                     
            // Get a command from the keyboard
            pc.printf("Enter command: ");         
            while (TRUE)
                {
                prevCharIn = charIn;
                charIn = pc.getc();
                pc.printf("%c", charIn);
                if ((charIn == '\n') || (charIn == '\r'))
                    {
                    break;
                    }
                }
            Command=prevCharIn;
                
            // Action the command
            switch(Command) 
                {
                case '1':      
                    pc.printf("Starting monitor\r\n");
                    Sample_index=0;
                    monitoring=TRUE;
                    monitor_tick.attach(&monitor_accelleration,MONITORINTERVAL);
                break;
                
                case '2':      
                    pc.printf("Stopping monitoring\r\n");
                    monitoring=FALSE;
                    LED=FALSE;
                    monitor_tick.detach();
                    Last_sample=Sample_index;
                break;
                
                case '3':
                    if (monitoring==TRUE)
                        {
                        pc.printf("Stopping monitoring\r\n");
                        monitoring=FALSE;
                        LED=FALSE;
                        monitor_tick.detach();
                        Last_sample=Sample_index;    
                        }
                    
                    if (Last_sample>0)
                        {
                        for (i=0; i<Last_sample; i++)
                            {
                            pc.printf("No.: %d ",i);     
                            pc.printf("X: %d ",X_store[i]);  
                            pc.printf("Y: %d ",Y_store[i]);    
                            pc.printf("Z: %d \r\n",Z_store[i]);        
                            }
                        }
                break;
                
                case '4':
                    pc.printf("Dumping EEPROM\r\n");     
                    // Read EEPROM into Storage arrays
                    IAP_Eeprom_Read(EEPROMBASEADDRESS, (uint8_t *)&X_store, sizeof(X_store));                                   // Read EEPROM into X array       
                    IAP_Eeprom_Read(EEPROMBASEADDRESS+sizeof(X_store), (uint8_t *)&Y_store, sizeof(Y_store));                   // Read EEPROM into Y array       
                    IAP_Eeprom_Read(EEPROMBASEADDRESS+sizeof(X_store)+sizeof(Y_store), (uint8_t *)&Z_store, sizeof(Z_store));   // Read EEPROM into Z array       
                    // Display them
                    for (i=0; i<MAXSAMPLES; i++)
                        {
                        pc.printf("No.: %d ",i);     
                        pc.printf("X: %d ",X_store[i]);  
                        pc.printf("Y: %d ",Y_store[i]);    
                        pc.printf("Z: %d \r\n",Z_store[i]);        
                        }
                break;
                
                case '9':
                    pc.printf("Clearing EEPROM\r\n");     
                    // Fill one of the arrays with zeros
                    for (i=0; i<MAXSAMPLES; i++)
                        {
                        X_store[i]=(uint16_t)j;    
                        }
                    // Write this array to X, Y & Z EEPROM stores
                    IAP_Eeprom_Write(EEPROMBASEADDRESS, (uint8_t *)&X_store, sizeof(X_store));              
                    IAP_Eeprom_Write(EEPROMBASEADDRESS+sizeof(X_store), (uint8_t *)&X_store, sizeof(X_store));            
                    IAP_Eeprom_Write(EEPROMBASEADDRESS+sizeof(X_store)+sizeof(Y_store), (uint8_t *)&X_store, sizeof(X_store));            
                break;
                
                default:
                    pc.printf("Invalid command\r\n");
                break;
                }
            }
        }

}

