/**
   @file
   @brief Example of using the XBeeApi library to read inputs on
          a remote XBee

   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#include "mbed.h"   
#include "xbeeapi.hpp"

Serial pc(USBTX, USBRX); // tx, rx

/* TODO: You may need to change these based on the device/connections that you're using */ 
#define XBEE_TX_PIN PTA2
#define XBEE_RX_PIN PTA1

/* Network address for our XBee */
const uint16_t myNetworkAddress = 0x1234;

/* Network address for the remote XBee */
const uint16_t remoteNetworkAddress = 0x4321;

/* ID for the Personal Area Network we're going to join */
const XBeeApiCmdAt::panId_t myPANId = 1000;

/* Network channel to use */
const XBeeApiCmdAt::channel_t myChannelId = 14;

XBeeDevice xbeeDevice( XBEE_TX_PIN, XBEE_RX_PIN, NC, NC );

/* Once the remote XBee is set up this function will monitor for changes to
   the I/O readings and report them */
void monitor( XBeeDeviceRemoteAt& p_remote )
{
    time_t   analogueTime[ XBEE_API_ADC_CHANNEL_COUNT ] = { 0, };
    time_t   digitalTime[ XBEE_API_DIO_CHANNEL_COUNT ] = { 0, };

    while(1)
    {
        uint8_t i;
        
        /* Loop all the analogue channels, see if the timestamp
           associated with the value has changed and if so, 
           report it */
        for( i = 0;
             i < XBEE_API_ADC_CHANNEL_COUNT;
             i++ )
        {
            uint16_t val;
            time_t updateTime = p_remote.getAnalogueValue( i, val );
            
            if( updateTime != analogueTime[ i ] )
            {
                analogueTime[i] = updateTime;
                pc.printf("Analogue Channel %d updated : %d\r\n",i,val);
            }
        }
        
        /* Loop all the digital channels, report changes */
        for( i = 0;
             i < XBEE_API_DIO_CHANNEL_COUNT;
             i++ )
        {
            bool val;
            time_t updateTime = p_remote.getDigitalState( i, val );

            if( updateTime != digitalTime[ i ] )
            {
                digitalTime[i] = updateTime;
                pc.printf("Digital Channel %d updated : %d\r\n",i,val);
            }
        }
        
        wait_ms(1000);
    }
}

int main() {
    /* This example will use the blocking API for simplicity */   
    XBeeApiCmdAtBlocking atIf( &xbeeDevice );

    XBeeDevice::XBeeDeviceReturn_t status;

    /* Get API mode 2 set up - this is a pre-requisit to using other XBeeApi functions.
       This step may not be needed in the case that the XBee has already been configured
       to use Mode 2 and the setting has been stored in NV */
    status = xbeeDevice.setUpApi();

    if( status == XBeeDevice::XBEEDEVICE_OK )
    {
        uint16_t readNetworkAddress;
        
        /* Set the 16-bit source address of this XBee */
        atIf.setSourceAddress( myNetworkAddress );

        if( atIf.getSourceAddress( &readNetworkAddress ) )
        {
            /* Set up a peer-to-peer network using the specified PAN and channel */
            if( xbeeSetNetworkTypeP2P( &atIf, myPANId, myChannelId ) )
            {
                uint16_t sampleRate;
                XBeeApiCmdAt::XBeeApiDioConfig_e dioConfig;
                uint32_t remoteDaHigh,remoteDaLow;
                /* Set up a remote XBee device using the specified 16-bit address (64-bit address not used here) */
                XBeeDeviceRemoteAt remoteXBee( &xbeeDevice, remoteNetworkAddress ); 
                
                /* Tell the destination XBee that it should transmit packets to this XBee's address */
                remoteXBee.setDestinationAddress( myNetworkAddress );
                
                /* Set I/O port 0 on remote XBee to be a digital input */
                remoteXBee.setDioConfig( 0, XBeeApiCmdAt::XBEE_API_DIO_INPUT );

                /* Set I/O port 1 on remote XBee to be an analogue input */
                remoteXBee.setDioConfig( 1, XBeeApiCmdAt::XBEE_API_DIO_ADC );
                
                /* Set sample rate on remote XBee to 100ms */
                remoteXBee.setSampleRate( 100 );
                
                /* Apply the above changes to the remote XBee.  Calling remoteXBee.setApplyChanges(true) before
                   the setDestinationAddress() would apply the changes as they were made and mean that
                   this call would not be necessary */
                remoteXBee.requestApplyChanges();
                
                wait_ms( 500 );
                
                if( !remoteXBee.getSampleRate( &sampleRate ) )
                {
                    pc.printf("Didn't get sample rate\r\n");
                }            
                if( !remoteXBee.getDioConfig( 0, &dioConfig ) )
                {
                    pc.printf("Didn't get dio config\r\n");
                }            
                if( !remoteXBee.getDestinationAddressHigh( &remoteDaHigh ) )
                {
                    pc.printf("Didn't get destination address (high)\r\n");
                }            
                if( !remoteXBee.getDestinationAddressLow( &remoteDaLow ) )
                {
                    pc.printf("Didn't get destination address (low)\r\n");
                }            
    
                monitor( remoteXBee );
            }  
            else
            {
                pc.printf("xbeeSetNetworkTypeP2P failed\r\n");
            }
        } 
        else
        {
            pc.printf("setSourceAddress() failed\r\n");
        }
    }
    else
    {
        pc.printf("setUpApi failed with status %d\r\n",status);
    }
}