/**
   @file
   @brief Example of using the XBeeApi library to send AT commands
          to remote XBee devices in order to read/write settings

   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#include "mbed.h"   
#include "xbeeapi.hpp"

Serial pc(USBTX, USBRX); // tx, rx

/* TODO: You may need to change these based on the device/connections that you're using */ 
#define XBEE_TX_PIN PTA2
#define XBEE_RX_PIN PTA1

/* Network address for our XBee */
const uint16_t myNetworkAddress = 0x1234;

/* Network address for the remote XBee */
const uint16_t remoteNetworkAddress = 0x4321;

/* ID for the Personal Area Network we're going to join */
const XBeeApiCmdAt::panId_t myPANId = 1000;

/* Network channel to use */
const XBeeApiCmdAt::channel_t myChannelId = 14;

XBeeDevice xbeeDevice( XBEE_TX_PIN, XBEE_RX_PIN, NC, NC );

int main() {
    /* This example will use the blocking API for simplicity */   
    XBeeApiCmdAtBlocking atIf( &xbeeDevice );

    XBeeDevice::XBeeDeviceReturn_t status;

    /* Get API mode 2 set up - this is a pre-requisit to using other XBeeApi functions.
       This step may not be needed in the case that the XBee has already been configured
       to use Mode 2 and the setting has been stored in NV */
    status = xbeeDevice.setUpApi();

    if( status == XBeeDevice::XBEEDEVICE_OK )
    {
        /* Set the 16-bit source address of this XBee */
        atIf.setSourceAddress( myNetworkAddress );

        /* Set up a peer-to-peer network using the specified PAN and channel */
        if( xbeeSetNetworkTypeP2P( &atIf, myPANId, myChannelId ) )
        {
            uint64_t sn;
            XBeeApiCmdAt::panId_t panId;
            uint8_t chan;
            
            /* Set up a remote XBee device using the specified 16-bit address (64-bit address not used here) */
            XBeeDeviceRemoteAt remoteXBee( &xbeeDevice, remoteNetworkAddress, 0 ); 
            
            /* Get some settings for the remote XBee */
            remoteXBee.requestPanId();
            remoteXBee.requestSerialNumber();
            remoteXBee.requestChannel();
            
            /* Get some seetings for the local XBee */
            atIf.requestPanId();
            atIf.requestSerialNumber();
            atIf.requestChannel();
            
            /* TODO: Add example of changing settings */
            
            wait_ms( 1000 );
            
            if( remoteXBee.getSerialNumber( &sn ) )
            {
                pc.printf("Remote serial number is %llu (decimal)\r\n",sn);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the remote serial number\r\n");
            }            
            if( remoteXBee.getPanId( &panId ) )
            {
                pc.printf("Remote PAN ID is %d\r\n",panId);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the remote PAN Id\r\n");
            }            
            
            if( remoteXBee.getChannel( &chan ) )
            {
                pc.printf("Local channel is %d\r\n",chan);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the local channel\r\n");
            }            
            if( atIf.getSerialNumber( &sn ) )
            {
                pc.printf("Local serial number is %llu (decimal)\r\n",sn);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the local serial number\r\n");
            }            
            if( atIf.getPanId( &panId ) )
            {
                pc.printf("Remote PAN ID is %d\r\n",panId);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the local PAN Id\r\n");
            }            
            if( atIf.getChannel( &chan ) )
            {
                pc.printf("Local channel is %d\r\n",chan);
            }
            else
            {
                pc.printf("Didn't manage to retrieve the local channel\r\n");
            }            
        }  
        else
        {
            pc.printf("xbeeSetNetworkTypeP2P failed\r\n");
        }
    }
    else
    {
        pc.printf("setUpApi failed with status %d\r\n",status);
    }
}