/**
   @file
   @brief Class to abstract AT commands send to the XBee API
          using a blocking API.
   
   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#if !defined XBEEAPICMDATBLOCKING_HPP
#define      XBEEAPICMDATBLOCKING_HPP

#include "XBeeApiCmdAt.hpp"

#include <stdint.h>

/** Class to access the configuration interface of the XBee.
    In contrast to XBeeApiCmdAt, the getXXX methods block
    until the data is received (or a timeout has occurred)
    which means that the caller doesn't have to deal with the
    asynchronous nature of the API provided by XBeeApiCmdAt.
    
    It's not necessary to use any of the requestXXX methods
    (as the getXXX methods will take care of this, however
    calling a requestXXX method will effectively pre-fetch the
    data meaning that getXXX will not have to block */
class XBeeApiCmdAtBlocking : public XBeeApiCmdAt
{
    protected:
        /** Timeout used for blocking methods in milliseconds */
        uint16_t m_timeout;
        
        /** Wait slice time while blocking.  The function will 
            wait_ms(m_slice) until the XBee responds with the
            data or m_timeout elapses */
        uint16_t m_slice;

    public:
       /** Constructor 
       
            \param p_device XBee device with which this object should 
                            be associated 
            \param p_timeout Timeout to be used when waiting for 
                             data from the XBee, specified in
                             milliseconds
            \param p_slice While waiting for data, blocking methods
                           will call the OS wait_ms() function, using
                           the value specified by p_slice */
       XBeeApiCmdAtBlocking( XBeeDevice* const p_device = NULL,
                            const uint16_t p_timeout = 1000, 
                            const uint16_t p_slice = 100);
       
       /** Destructor */
       virtual ~XBeeApiCmdAtBlocking( void ) {};
 
       /* Implement XBeeApiCmdAt's virtual methods */
       
        virtual bool getHardwareVersion( uint16_t* const p_ver );
        virtual bool getFirmwareVersion( uint16_t* const p_ver );
        virtual bool getSerialNumber( uint64_t* const p_sn );

        virtual bool getChannel( uint8_t* const p_chan );
        virtual bool setChannel( uint8_t const p_chan );
        
        virtual bool getCoordinatorEnabled( bool* constp_en );
        virtual bool setCoordinatorEnabled( const bool p_en );
       
        virtual bool getEndDeviceAssociationEnabled( bool* const p_en ); 
        virtual bool setEndDeviceAssociationEnabled( const bool p_en  );
       
        virtual bool getPanId( panId_t* const p_id );       
        virtual bool setPanId( const panId_t p_id );

        virtual bool getSourceAddress( uint16_t* const p_addr );       
        virtual bool setSourceAddress( const uint16_t p_addr );  
       
        virtual bool getRetries( uint8_t* const p_addr );       
        virtual bool setRetries( const uint8_t p_addr );       

        virtual bool getRandomDelaySlots( uint8_t* const p_addr );       
        virtual bool setRandomDelaySlots( const uint8_t p_addr );       

        virtual bool getMacMode( XBeeApiMACMode_e* const p_mode );       
        virtual bool setMacMode( const XBeeApiMACMode_e p_mode );  
        
        virtual bool getDioConfig( const uint8_t p_chanNo, XBeeApiDioConfig_e* const p_conf );
        virtual bool setDioConfig( const uint8_t p_chanNo, const XBeeApiDioConfig_e p_conf );
     
        virtual bool getDioChangeDetectMask( uint8_t* const p_mask );
        virtual bool setDioChangeDetectMask( const uint8_t p_mask );

        virtual bool getDioLevels( uint8_t* const p_mask );
        virtual bool setDioLevels( const uint8_t p_mask );

        virtual bool getSampleRate( uint16_t* const p_interval );
        virtual bool setSampleRate( const uint16_t p_interval );

        virtual bool getDestinationAddress( uint64_t* const p_address );
        virtual bool setDestinationAddress( const uint64_t p_address );

        virtual bool getDestinationAddressHigh( uint32_t* const p_address );
        virtual bool setDestinationAddressHigh( const uint32_t p_address );

        virtual bool getDestinationAddressLow( uint32_t* const p_address );
        virtual bool setDestinationAddressLow( const uint32_t p_address );
        
        virtual bool setEncryptionKey( const uint8_t* p_key );
        
        virtual bool getEncryptionEnabled( bool* const p_enabled );
        virtual bool setEncryptionEnabled( const bool  p_enabled );
};

#endif