/**
   @file
   @brief Class to provide an abstract representation of a remote
          XBee device's AT command interface
      
   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#if !defined XBEEDEVICEREMOTEAT_HPP
#define      XBEEDEVICEREMOTEAT_HPP

#include "XBeeApiCfg.hpp"
#include "XBeeApiFrame.hpp"
#include "XBeeApiCmdAt.hpp"

/** Class used to provide a command interface to a remove XBee via the Remote AT interface.
    The class inherits from XBeeApiCmdAt, hence the interface for the local and remote
    XBees is generally the same.
*/
class XBeeDeviceRemoteAt : public XBeeApiCmdAt
{
    protected:
        /** Called by XBeeDevice in order to offer frame data to the object for
            decoding
           
            \param p_data Pointer to the content of the received data
            \param p_len Length of the data pointed to by p_data
        */
        virtual bool decodeCallback( const uint8_t* const p_data, size_t p_len );

        virtual void SendCmd( const uint8_t        p_frameId,
                              const uint8_t* const p_data,
                              const uint8_t* const p_val,
                              const uint8_t        p_len );

        /* TODO: doc */
        virtual size_t getResponseStatusPos( void ) const;

        /** Keep track of whether we're applying changes when the AT command is sent
            or not */
        bool     m_applyChanges;

        /** Keep track of which type of addressing we're using for transmitted frames */
        XBeeDevice::XBeeApiAddrType_t m_addressingType;

        /** Set m_snLow/m_snHigh or m_sourceAddress depending on the parameters specified and
            set up suplementary attributes such as the addressing type to be used
            
            \param p_addr16Bit 16-bit address of the remote XBee (or XBEE_BROADCAST_ADDR for broadcast or XBEE_USE_64BIT_ADDR to use 64-bit addressing (see p_addr64Bit)
            \param p_addr64Bit 64-bit address of the remote XBee, only used in the case that the 16-bit address is XBEE_USE_64BIT_ADDR
        */
        void setAddress( const uint16_t& p_addr16Bit,
                         const uint64_t& p_addr64Bit = 0 );

    public:
    
        /** Constructor 
        
            \param p_device Pointer to the XBee device used to communicate with the remote XBee 
            \param p_addr16Bit 16-bit address of the remote XBee (or XBEE_BROADCAST_ADDR for broadcast or XBEE_USE_64BIT_ADDR to use 64-bit addressing (see p_addr64Bit)
            \param p_addr64Bit 64-bit address of the remote XBee, only used in the case that the 16-bit address is XBEE_USE_64BIT_ADDR
            \param p_applyChanges Indicate whether or not changes should be applied with each message/method call.  Seed setApplyChanges() */
        XBeeDeviceRemoteAt( XBeeDevice* p_device,
                            const uint16_t& p_addr16Bit,
                            const uint64_t& p_addr64Bit = 0,
                            const bool      p_applyChanges = false );
        
        /** Destructor */
        virtual ~XBeeDeviceRemoteAt( void ); 
        
        /** Control whether or not changes are applied when the command is sent or not.
            There are three possibilities for appling changes on the remote:
            1) Use this method, specifying a 'true' parameter so that changes are applied as soon as a command is sent
            2) Use requestApplyChanges() to apply all pending changes
            3) Use requestWriteSettings() followed by requestReset() to write the settings to non-volatile memory and reset the device
        
            \param Specified whether or not the settings should be applied when the command is sent to the remote device */
        void setApplyChanges( const bool p_apply );

        /** Set the type of addressing used for transmitted frames to the remote XBee.  If specifying 16-bit addressing then the
            remote node's 16-bit address must be known.  If specifying 64-bit addressing then the remote node's 64-bit address must
            be known.  The addresses are learned either by specifying them to the constructor or by successful requests via 
            requestSourceAddress (16-bit) or requestSerialNumber (64-bit), or by using reassociate()
         
            \param p_type The type of addressing to use for future transmitted frames
            \returns true in the case that the address type was succesfully changed,
                     false in the case that the address type could not be changed, either because p_type was not valid
                           or because the appropriate address wasn't known */
        bool setAddressingType( const XBeeDevice::XBeeApiAddrType_t p_type );
        
        /** Reassociated this object with a new remote XBee (or update the addressing used).  Calling this method will reset
            any cached values held within the object associated with the remote XBee's settings.
            
            \param p_addr16Bit 16-bit address of the remote XBee (or XBEE_BROADCAST_ADDR for broadcast or XBEE_USE_64BIT_ADDR to use 64-bit addressing (see p_addr64Bit)
            \param p_addr64Bit 64-bit address of the remote XBee, only used in the case that the 16-bit address is XBEE_USE_64BIT_ADDR
        */
        void reassociate( const uint16_t& p_addr16Bit,
                          const uint64_t& p_addr64Bit = 0 );              
};

#endif