/**

Copyright 2014 John Bailey

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#include "XBeeApiCmdAtBlocking.hpp"

XBeeApiCmdAtBlocking::XBeeApiCmdAtBlocking( XBeeDevice* const p_device, const uint16_t p_timeout, const uint16_t p_slice ) :
    XBeeApiCmdAt( p_device ),
    m_timeout( p_timeout ),
    m_slice( p_slice )
{
}

/**
   Macro to wrap around the "requestXXX" & "getXXX" methods and implement a blocking call.
   This macro is used as the basis for getXXX functions in XBeeApiCmdAtBlocking.

   Originally looked to achieve this using a template function passing method pointers, however
   there's no way to get a method pointer to the parent class implementation as opposed to the
   implementation in this class, meaning that the result was a recursive method call.  The joys of
   polymorphism.

   e.g. We pass a pointer to method getHardwareVersion().  The function receiving the pointer
        uses it to make a function call.  The actual function that's called is (correctly)
        the one implemented in this class, however what we actually wanted in this case
        was to call the implementation in the base class.  Using static_cast<> doesn't have
        any effect and taking the address of XBeeApiCmdAt::getHardwareVersion ends up with
        XBeeApiCmdAtBlocking::getHardwareVersion being called due to polymorphism. */
#define BLOCKING_GET( _REQ_FN, _GET_FN, _VAR ) \
    bool ret_val = false; \
\
    if( _GET_FN( _VAR ) )\
    {\
        ret_val = true;\
    } \
    else if( _REQ_FN() )\
    {\
        uint16_t counter = m_timeout; \
        bool cont;\
\
        do{\
            cont = false; \
            wait_ms( m_slice );\
            if( _GET_FN( _VAR ) )\
            {\
                ret_val = true;\
            }\
            else if( counter > m_slice ) {\
                counter -= m_slice;    \
                cont = true;\
            } \
        } while( cont );\
    }\
    \
    return( ret_val );

/**
   Macro to wrap around the "setXXX" & "getXXX" methods and implement a blocking call.
   This macro is used as the basis for setXXX functions in XBeeApiCmdAtBlocking.
*/
#define BLOCKING_SET( _SET_FN, _GET_FN, _VAR, _TYPE ) \
    bool ret_val = false; \
    uint16_t counter = m_timeout; \
    _TYPE readback; \
\
    if( _SET_FN( _VAR ) )\
    {\
        bool cont;\
\
        do{\
            cont = false;\
            wait_ms( m_slice );\
            if( _GET_FN( &readback ) &&\
               ( readback == _VAR ))\
            {\
                ret_val = true;\
            }\
            else if( counter > m_slice ) {\
                counter -= m_slice;    \
                cont = true;\
            } \
        } while( cont );\
    }\
    \
    return( ret_val );


bool XBeeApiCmdAtBlocking::getHardwareVersion( uint16_t* const p_ver )
{
    BLOCKING_GET( XBeeApiCmdAt::requestHardwareVersion,
                  XBeeApiCmdAt::getHardwareVersion,
                  p_ver );
}

bool XBeeApiCmdAtBlocking::getFirmwareVersion( uint16_t* const p_ver )
{
    BLOCKING_GET( XBeeApiCmdAt::requestFirmwareVersion,
                  XBeeApiCmdAt::getFirmwareVersion,
                  p_ver );
}

bool XBeeApiCmdAtBlocking::getSerialNumber( uint64_t* const p_sn )
{
    BLOCKING_GET( XBeeApiCmdAt::requestSerialNumber,
                  XBeeApiCmdAt::getSerialNumber,
                  p_sn );
}

bool XBeeApiCmdAtBlocking::getChannel( uint8_t* const p_chan )
{
    BLOCKING_GET( XBeeApiCmdAt::requestChannel,
                  XBeeApiCmdAt::getChannel,
                  p_chan );
}

bool XBeeApiCmdAtBlocking::setChannel( uint8_t const p_chan )
{
    BLOCKING_SET( XBeeApiCmdAt::setChannel,
                  XBeeApiCmdAt::getChannel,
                  p_chan,
                  uint8_t );
}

bool XBeeApiCmdAtBlocking::getCoordinatorEnabled( bool* const p_en )
{
    BLOCKING_GET( XBeeApiCmdAt::requestCoordinatorEnabled,
                  XBeeApiCmdAt::getCoordinatorEnabled,
                  p_en );
}

bool XBeeApiCmdAtBlocking::setCoordinatorEnabled( const bool p_en )
{
    BLOCKING_SET( XBeeApiCmdAt::setCoordinatorEnabled,
                  XBeeApiCmdAt::getCoordinatorEnabled,
                  p_en,
                  bool );
}

bool XBeeApiCmdAtBlocking::getEndDeviceAssociationEnabled( bool* const p_en )
{
    BLOCKING_GET( XBeeApiCmdAt::requestEndDeviceAssociationEnabled,
                  XBeeApiCmdAt::getEndDeviceAssociationEnabled,
                  p_en );
}

bool XBeeApiCmdAtBlocking::setEndDeviceAssociationEnabled( const bool p_en )
{
    BLOCKING_SET( XBeeApiCmdAt::setEndDeviceAssociationEnabled,
                  XBeeApiCmdAt::getEndDeviceAssociationEnabled,
                  p_en,
                  bool );
}

bool XBeeApiCmdAtBlocking::getPanId( panId_t* const p_chan )
{
    BLOCKING_GET( XBeeApiCmdAt::requestPanId,
                  XBeeApiCmdAt::getPanId,
                  p_chan );
}

bool XBeeApiCmdAtBlocking::setPanId( const panId_t p_chan )
{
    BLOCKING_SET( XBeeApiCmdAt::setPanId,
                  XBeeApiCmdAt::getPanId,
                  p_chan,
                  panId_t );
}

bool XBeeApiCmdAtBlocking::getSourceAddress( uint16_t* const p_addr )
{
    BLOCKING_GET( XBeeApiCmdAt::requestSourceAddress,
                  XBeeApiCmdAt::getSourceAddress,
                  p_addr );
}

bool XBeeApiCmdAtBlocking::setSourceAddress( const uint16_t p_addr )
{
    BLOCKING_SET( XBeeApiCmdAt::setSourceAddress,
                  XBeeApiCmdAt::getSourceAddress,
                  p_addr,
                  uint16_t );
}

bool XBeeApiCmdAtBlocking::getRetries( uint8_t* const p_addr )
{
    BLOCKING_GET( XBeeApiCmdAt::requestRetries,
                  XBeeApiCmdAt::getRetries,
                  p_addr );
}

bool XBeeApiCmdAtBlocking::setRetries( const uint8_t p_addr )
{
    BLOCKING_SET( XBeeApiCmdAt::setRetries,
                  XBeeApiCmdAt::getRetries,
                  p_addr,
                  uint8_t );
}

bool XBeeApiCmdAtBlocking::getRandomDelaySlots( uint8_t* const p_addr )
{
    BLOCKING_GET( XBeeApiCmdAt::requestRandomDelaySlots,
                  XBeeApiCmdAt::getRandomDelaySlots,
                  p_addr );
}

bool XBeeApiCmdAtBlocking::setRandomDelaySlots( const uint8_t p_addr )
{
    BLOCKING_SET( XBeeApiCmdAt::setRandomDelaySlots,
                  XBeeApiCmdAt::getRandomDelaySlots,
                  p_addr,
                  uint8_t );
}

bool XBeeApiCmdAtBlocking::getMacMode( XBeeApiMACMode_e* const p_mode )
{
    BLOCKING_GET( XBeeApiCmdAt::requestMacMode,
                  XBeeApiCmdAt::getMacMode,
                  p_mode );
}
           
bool XBeeApiCmdAtBlocking::setMacMode( const XBeeApiMACMode_e p_mode )
{
    BLOCKING_SET( XBeeApiCmdAt::setMacMode,
                  XBeeApiCmdAt::getMacMode,
                  p_mode,
                  XBeeApiMACMode_e );
}

bool XBeeApiCmdAtBlocking::getDioConfig( const uint8_t p_chanNo, XBeeApiDioConfig_e* const p_conf )
{
    /* TODO */
    return false;
}

bool XBeeApiCmdAtBlocking::setDioConfig( const uint8_t p_chanNo, const XBeeApiDioConfig_e p_conf )
{
    /* TODO */
    return false;
}

bool XBeeApiCmdAtBlocking::getDioChangeDetectMask( uint8_t* const p_mask )
{
    BLOCKING_GET( XBeeApiCmdAt::requestDioChangeDetectMask,
                  XBeeApiCmdAt::getDioChangeDetectMask,
                  p_mask );
}

bool XBeeApiCmdAtBlocking::setDioChangeDetectMask( const uint8_t p_mask )
{
    BLOCKING_SET( XBeeApiCmdAt::setDioChangeDetectMask,
                  XBeeApiCmdAt::getDioChangeDetectMask,
                  p_mask,
                  uint8_t );
}

bool XBeeApiCmdAtBlocking::getDioLevels( uint8_t* const p_mask )
{
/* TODO */
return false;
}

bool XBeeApiCmdAtBlocking::setDioLevels( const uint8_t p_mask )
{
    BLOCKING_SET( XBeeApiCmdAt::setDioLevels,
                  XBeeApiCmdAt::getDioLevels,
                  p_mask,
                  uint8_t );
}

bool XBeeApiCmdAtBlocking::getSampleRate( uint16_t* const p_interval )
{
    BLOCKING_GET( XBeeApiCmdAt::requestSampleRate,
                  XBeeApiCmdAt::getSampleRate,
                  p_interval );
}

bool XBeeApiCmdAtBlocking::setSampleRate( const uint16_t p_interval )
{
    BLOCKING_SET( XBeeApiCmdAt::setSampleRate,
                  XBeeApiCmdAt::getSampleRate,
                  p_interval,
                  uint16_t );
}

bool XBeeApiCmdAtBlocking::getDestinationAddress( uint64_t* const p_address )
{
    BLOCKING_GET( XBeeApiCmdAt::requestDestinationAddress,
                  XBeeApiCmdAt::getDestinationAddress,
                  p_address );
}

bool XBeeApiCmdAtBlocking::setDestinationAddress( const uint64_t p_address )
{
    BLOCKING_SET( XBeeApiCmdAt::setDestinationAddress,
                  XBeeApiCmdAt::getDestinationAddress,
                  p_address,
                  uint64_t );
}

bool XBeeApiCmdAtBlocking::getDestinationAddressHigh( uint32_t* const p_address )
{
    BLOCKING_GET( XBeeApiCmdAt::requestDestinationAddressHigh,
                  XBeeApiCmdAt::getDestinationAddressHigh,
                  p_address );
}    
    
bool XBeeApiCmdAtBlocking::setDestinationAddressHigh( const uint32_t p_address )
{
    BLOCKING_SET( XBeeApiCmdAt::setDestinationAddressHigh,
                  XBeeApiCmdAt::getDestinationAddressHigh,
                  p_address,
                  uint32_t );
}

bool XBeeApiCmdAtBlocking::getDestinationAddressLow( uint32_t* const p_address )
{
    BLOCKING_GET( XBeeApiCmdAt::requestDestinationAddressLow,
                  XBeeApiCmdAt::getDestinationAddressLow,
                  p_address );
}

bool XBeeApiCmdAtBlocking::setDestinationAddressLow( const uint32_t p_address )
{
    BLOCKING_SET( XBeeApiCmdAt::setDestinationAddressLow,
                  XBeeApiCmdAt::getDestinationAddressLow,
                  p_address,
                  uint32_t );
}

bool XBeeApiCmdAtBlocking::setEncryptionKey( const uint8_t* p_key )
{
    bool ret_val = false;
    uint16_t counter = m_timeout; 

    if( XBeeApiCmdAt::setEncryptionKey( p_key ) )\
    {
        bool cont;

        do{
            cont = false;
            wait_ms( m_slice );
            if( XBeeApiCmdAt::getEncryptionKeySet() )
            {
                ret_val = true;
            }
            else if( counter > m_slice ) {
                counter -= m_slice;    
                cont = true;
            } 
        } while( cont );
    }
    
    return( ret_val );
}
        
bool XBeeApiCmdAtBlocking::getEncryptionEnabled( bool* const p_enabled )
{
    BLOCKING_GET( XBeeApiCmdAt::requestEncryptionEnabled,
                  XBeeApiCmdAt::getEncryptionEnabled,
                  p_enabled );
}

bool XBeeApiCmdAtBlocking::setEncryptionEnabled( const bool  p_enabled )
{
    BLOCKING_SET( XBeeApiCmdAt::setEncryptionEnabled,
                  XBeeApiCmdAt::getEncryptionEnabled,
                  p_enabled,
                  bool );
}
