/** 

Copyright 2014 John Bailey
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#include "XBeeDeviceRemoteAt.hpp"

#define XBEE_API_CMD_REMOTE_REQ_HEADER_LEN 14U

class XBeeApiCmdAtRemoteFrame : public XBeeApiFrame {
    uint8_t m_buffer[ XBEE_API_CMD_REMOTE_REQ_HEADER_LEN + XBEE_CMD_MAX_PARAM_LENGTH ];
    public:
        /** Constructor
                   
            \param p_data Pointer to a buffer of length 2 bytes identifying
                          the command, e.g. 'V', 'R' would set up a version
                          request
            \param p_val New value for the parameter 
        */
        XBeeApiCmdAtRemoteFrame( const uint8_t        p_frameId,
                                 const uint16_t       p_addr16Bit,
                                 const uint64_t       p_addr64Bit,
                                 const XBeeDevice::XBeeApiAddrType_t p_type,
                                 const bool           p_applyChanges,
                                 const uint8_t* const p_data,
                                 const uint8_t* const p_val,
                                 const uint8_t        p_len );
        /** Destructor */
        virtual ~XBeeApiCmdAtRemoteFrame();
};

XBeeDeviceRemoteAt::XBeeDeviceRemoteAt( XBeeDevice* p_device,
                                        const uint16_t& p_addr16Bit,
                                        const uint64_t& p_addr64Bit,
                                        const bool      p_applyChanges  ) : XBeeApiCmdAt( p_device ), m_applyChanges( p_applyChanges )
{
    setAddress( p_addr16Bit, p_addr64Bit );
}

void XBeeDeviceRemoteAt::setAddress( const uint16_t& p_addr16Bit,
                                     const uint64_t& p_addr64Bit )
{
    if( p_addr16Bit != XBEE_USE_64BIT_ADDR )
    {
        m_sourceAddress = p_addr16Bit;
        m_have_sourceAddress = true;
        m_snLow = m_snHigh = 0;
        m_addressingType = XBeeDevice::XBEE_API_ADDR_TYPE_16BIT;
    } else
    {
        m_snLow = (p_addr64Bit & 0xFFFFFFFF);
        m_snHigh = ((p_addr64Bit >> 32U) & 0xFFFFFFFF);
        m_sourceAddress = XBEE_USE_64BIT_ADDR;
        m_have_snLow = m_have_snHigh = true;
        m_addressingType = XBeeDevice::XBEE_API_ADDR_TYPE_64BIT;
    }  
}
       
void XBeeDeviceRemoteAt::reassociate( const uint16_t& p_addr16Bit,
                                      const uint64_t& p_addr64Bit )
{
    resetCachedData();
    setAddress( p_addr16Bit, p_addr64Bit );
}

        
XBeeDeviceRemoteAt::~XBeeDeviceRemoteAt( void )
{
}

void XBeeDeviceRemoteAt::setApplyChanges( const bool p_apply )
{
    m_applyChanges = p_apply;
}

#define XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS (XBEE_CMD_POSN_ID_SPECIFIC_DATA+1)
#define XBEE_REMOTE_AT_RESPONSE_16BIT_ADDRESS (XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + sizeof( uint64_t))
#define XBEE_REMOTE_AT_RESPONSE_STATUS (17U)

size_t XBeeDeviceRemoteAt::getResponseStatusPos( void ) const
{
    return XBEE_REMOTE_AT_RESPONSE_STATUS;
}

bool XBeeDeviceRemoteAt::decodeCallback( const uint8_t* const p_data, size_t p_len )
{
    bool ret_val = false;
 
    /* TODO: Length check */
 
    if( XBEE_CMD_REMOTE_AT_RESPONSE == p_data[ XBEE_CMD_POSN_API_ID ] )
    {
        uint32_t srcAddrHigh = (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS ]) << 24U) |
                               (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 1 ]) << 16U) |
                               (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 2 ]) << 8U) |
                               (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 3 ]));
        uint32_t srcAddrLow =  (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 4 ]) << 24U) |
                                (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 5 ]) << 16U) |
                                (((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 6 ]) << 8U) |
                                ((uint64_t)p_data[ XBEE_REMOTE_AT_RESPONSE_64BIT_ADDRESS + 7 ]);

        uint16_t src16BitAddr = (((uint16_t)p_data[ XBEE_REMOTE_AT_RESPONSE_16BIT_ADDRESS ]) << 8U) | 
                                p_data[ XBEE_REMOTE_AT_RESPONSE_16BIT_ADDRESS + 1 ];
                        
        if((( m_have_sourceAddress ) && ( m_sourceAddress == src16BitAddr )) ||
           ( m_have_snHigh && m_have_snLow && ( srcAddrHigh == m_snHigh ) && ( srcAddrLow == m_snLow )))
        {
            ret_val = processResponseFrame( p_data, p_len );
        }
    }
    else
    {
        ret_val = XBeeApiCmdAt::decodeCallback( p_data, p_len );
    }
    
    return ret_val;
}

void XBeeDeviceRemoteAt::SendCmd( const uint8_t p_frameId,
                                  const uint8_t* const p_data,
                                  const uint8_t* const p_val,
                                  const uint8_t        p_len )
{
     uint64_t addr64Bit = (((uint64_t)m_snHigh) << 32U) | (uint64_t)m_snLow;
     /* TODO: Add option to force usage of 16 or 64-bit addressing */
     XBeeApiCmdAtRemoteFrame req( p_frameId, m_sourceAddress, addr64Bit, m_addressingType, m_applyChanges, p_data, p_val, p_len );
     m_device->SendFrame( &req );
}

bool XBeeDeviceRemoteAt::setAddressingType( const XBeeDevice::XBeeApiAddrType_t p_type )
{
    bool ret_val = false;
    
    switch( p_type )
    {
        case XBeeDevice::XBEE_API_ADDR_TYPE_16BIT:
            if( m_have_sourceAddress )
            {
                m_addressingType = XBeeDevice::XBEE_API_ADDR_TYPE_16BIT;
                ret_val = true;
            }
            break;
        case XBeeDevice::XBEE_API_ADDR_TYPE_64BIT:
            if( m_have_snLow && m_have_snHigh )
            {
                m_addressingType = XBeeDevice::XBEE_API_ADDR_TYPE_64BIT;
                ret_val = true;
            }
            break;
    }
    
    return ret_val;
}

static void writeAddressToBuffer( uint8_t* const p_buffer,
                                  const uint16_t       p_addr16Bit,
                                  const uint64_t       p_addr64Bit,
                                  const XBeeDevice::XBeeApiAddrType_t p_type )
{
    if( p_type == XBeeDevice::XBEE_API_ADDR_TYPE_64BIT )
    {
        p_buffer[0] = p_addr64Bit >> 56U;
        p_buffer[1] = p_addr64Bit >> 48U;
        p_buffer[2] = p_addr64Bit >> 40U;
        p_buffer[3] = p_addr64Bit >> 32U;
        p_buffer[4] = p_addr64Bit >> 24U;
        p_buffer[5] = p_addr64Bit >> 16U;
        p_buffer[6] = p_addr64Bit >> 8U;
        p_buffer[7] = p_addr64Bit;

        p_buffer[8] = (uint8_t)(XBEE_USE_64BIT_ADDR >> 8U);
        p_buffer[9] = (uint8_t)(XBEE_USE_64BIT_ADDR & 0xFF);
    }
    else
    {

        p_buffer[0] = 0;
        p_buffer[1] = 0;
        p_buffer[2] = 0;
        p_buffer[3] = 0;
        p_buffer[4] = 0;
        p_buffer[5] = 0;
        p_buffer[6] = 0;
        p_buffer[7] = 0;
        
        p_buffer[8] = p_addr16Bit >> 8U;
        p_buffer[9] = p_addr16Bit;
    }
}

XBeeApiCmdAtRemoteFrame::XBeeApiCmdAtRemoteFrame( const uint8_t        p_frameId,
                                                  const uint16_t       p_addr16Bit,
                                                  const uint64_t       p_addr64Bit,
                                                  const XBeeDevice::XBeeApiAddrType_t p_type,
                                                  const bool           p_applyChanges,
                                                  const uint8_t* const p_data,
                                                  const uint8_t* const p_val,
                                                  const uint8_t        p_len ) : XBeeApiFrame( )
{   
    m_apiId = XBEE_CMD_REMOTE_AT_CMD;
    
    m_buffer[0] = p_frameId;
    
    writeAddressToBuffer( &(m_buffer[1]), p_addr16Bit, p_addr64Bit, p_type );

    if( p_len )
    {
        m_buffer[11] = p_applyChanges << 1;
    }
    else
    {
        m_buffer[11] = 0;
    }
    
    m_buffer[12] = p_data[0];
    m_buffer[13] = p_data[1];
    
    m_data = m_buffer;
    m_dataLen = XBEE_API_CMD_REMOTE_REQ_HEADER_LEN;
     
    if(( p_val != NULL ) &&
       ( p_len <= XBEE_CMD_MAX_PARAM_LENGTH ))
    {
        size_t s = 0;
        uint8_t* dest = &( m_buffer[ XBEE_API_CMD_REMOTE_REQ_HEADER_LEN ] );
        const uint8_t* src = p_val;
        
        for( s = 0;
             s < p_len;
             s++, dest++, src++ ) {
             *dest = *src;         
        }
        m_dataLen += p_len;
    }
    
#if 0
    /* Debugging code */
    extern Serial pc;
    size_t x = 0;
    pc.printf("\r\n[%02X][%02X][%02X]",(m_dataLen>>8U)&0xFF,m_dataLen&0xFF,m_apiId);
    for( x  = 0; x < m_dataLen; x++ )
    {
        if(( x> 11 ) && ( x < 14 )) 
        { 
            pc.printf("%c",m_data[x]); 
        }
        pc.printf("[%02X]",m_data[x]);
    }
    pc.printf("\r\n");
#endif

}

XBeeApiCmdAtRemoteFrame::~XBeeApiCmdAtRemoteFrame()
{
}