/**
   @file
   @brief Class to abstract AT commands send to the XBee API
   
   AT commands have the payload:
   
     Byte 1     : Frame ID
     Byte 2 & 3 : AT command
     Byte 4-n   : Parameter Value 
   
   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#if !defined XBEEAPICMDAT_HPP
#define      XBEEAPICMDAT_HPP

#include "XBeeApiFrame.hpp"
#include "XBeeDevice.hpp"

#include <stdint.h>

#define XBEE_API_CMD_SET_HEADER_LEN 3U
#define XBEE_API_CMD_REQ_HEADER_LEN 3U

#define XBEE_API_DIO_CHANNEL_COUNT 8U
#define XBEE_API_ADC_CHANNEL_COUNT 6U

/** Class to access the configuration interface of the XBee.
    Requests to the XBee are non-blocking meaning that code
    which utilises this class must deal with the fact that
    there will be a delay between requesting the data from
    the XBee and the data being available via the API.  See
    XBeeApiCmdAtBlocking for a blocking version.
    
    Parameters from the XBee are cached in the object so
    subsequent requests do not need have the overhead of
    communication with the XBee
    
    The following table cross-references between the AT commands
    and the method(s) associated with them.
    
    AT Command | Method(s)
    -----------|-----------------------------------------------
    WR         | requestWriteSettings
    RE         | requestRestoreDefaults
    FR         | requestReset
    CH         | requestChannel, setChannel, getChannel
    ID         | requestPanId, setPanId, getPanId
    DH         | requestDestinationAddress, requestDestinationAddressHigh, setDestinationAddressHigh, getDestinationAddressHigh
    DL         | requestDestinationAddress, requestDestinationAddressLow, setDestinationAddressLow, getDestinationAddressLow
    MY         | requestSourceAddress, getSourceAddress, setSourceAddress
    SH         | requestSerialNumber, requestSerialNumberHigh, getSerialNumber
    SL         | requestSerialNumber, requestSerialNumberLow, getSerialNumber
    RR         |
    RN         |
    MM         | requestMacMode, setMacMode, getMacMode
    NI         |
    ND         |
    NT         |
    NO         |
    DN         |
    CE         | requestCoordinatorEnabled, setCoordinatorEnabled, getCoordinatorEnabled
    SC         |
    SD         |
    A1         |
    A2         |
    AI         |
    DA         |
    FP         |
    AS         |
    ED         |
    EE         | requestEncryptionEnabled, setEncryptionEnabled, getEncryptionEnabled
    KY         | setEncryptionKey, getEncryptionKeySet
    PL         |
    CA         |
    SM         |
    SO         |
    ST         |
    SP         |
    DP         |
    BD         |
    RO         |
    AP         |
    NB         |
    PR         |
    D8         | requestDioConfig, setDioConfig, getDioConfig
    D7         | requestDioConfig, setDioConfig, getDioConfig
    D6         | requestDioConfig, setDioConfig, getDioConfig
    D5         | requestDioConfig, setDioConfig, getDioConfig
    D4         | requestDioConfig, setDioConfig, getDioConfig
    D3         | requestDioConfig, setDioConfig, getDioConfig
    D2         | requestDioConfig, setDioConfig, getDioConfig
    D1         | requestDioConfig, setDioConfig, getDioConfig
    D0         | requestDioConfig, setDioConfig, getDioConfig
    IU         |
    IT         |
    IS         | requestForceSample
    IO         |
    IC         | requestDioChangeDetectMask, setDioChangeDetectMask, getDioChangeDetectMask
    IR         | requestSampleRate, setSampleRate, getSampleRate
    IA         |
    T7         |
    T6         |
    T5         |
    T4         |
    T3         |
    T2         |
    T1         |
    T0         |
    P0         |
    P1         |
    M0         |
    M1         |
    PT         |
    RP         |
    VR         | requestFirmwareVersion, getFirmwareVersion
    VL         |
    HV         | requestHardwareVersion, getHardwareVersion
    EC         |
    EA         |
    ED         |
    CT         |
    CN         |
    AC         | requestApplyChanges
    GT         |
    CC         |
*/
class XBeeApiCmdAt : public XBeeApiFrameDecoder
{
    public:
        /** Type to represent the ID of a PAN (Personal Area Network) */
        typedef uint16_t panId_t;
        /** Type to represent a wireless channel number */
        typedef uint8_t  channel_t;
        /** Type to represent the different MAC modes supported by the XBee */
        typedef enum {
            XBEE_API_MAC_MODE_DIGI_ACK        = 0,
            XBEE_API_MAC_MODE_802_15_4_NO_ACK = 1,
            XBEE_API_MAC_MODE_802_15_4_ACK    = 2,
            XBEE_API_MAC_MODE_DIGI_NO_ACK     = 3,
        } XBeeApiMACMode_e;
    
        /** Type to represent the various different configuration options
            for the I/O pins.  Note that not all I/O pins will support all
            options */
        typedef enum
        {
            XBEE_API_DIO_DISABLED             = 0,
            XBEE_API_DIO_SPECIAL              = 1,
            XBEE_API_DIO_ADC                  = 2,
            XBEE_API_DIO_INPUT                = 3,
            XBEE_API_DIO_OUT_LOW              = 4,
            XBEE_API_DIO_OUT_HIGH             = 5,
            XBEE_API_DIO_RS485_TX_LOW         = 6,
            XBEE_API_DIO_RS485_TX_HIGH        = 7
        } XBeeApiDioConfig_e;
    
    protected:
        /** Indicates whether or not m_hwVer contains data retrieved from the XBee */
        bool m_have_hwVer;
        /** Indicates whether or not m_fwVer contains data retrieved from the XBee */
        bool m_have_fwVer;
        /** Indicates whether or not m_chan contains data retrieved from the XBee */
        bool m_have_chan;
        /** Indicates whether or not m_PANId contains data retrieved from the XBee */
        bool m_have_PANId;
        /** Indicates whether or not m_EDA contains data retrieved from the XBee */
        bool m_have_EDA;
        /** Indicates whether or not m_CE contains data retrieved from the XBee */
        bool m_have_CE;
        bool m_have_sourceAddress;
        bool m_have_snLow;
        bool m_have_snHigh;
        bool m_have_retries;
        bool m_have_randomDelaySlots;
        bool m_have_macMode;
        bool m_have_d[ XBEE_API_DIO_CHANNEL_COUNT ];
        bool m_have_dioChangeDetectMask;
        bool m_have_dioLevels;
        bool m_have_sampleRate;
        bool m_have_destHigh;
        bool m_have_destLow;
        bool m_have_encryptionEnabled;
        bool m_keySet;
 
        
        uint16_t m_hwVer;
        uint16_t m_fwVer;
        channel_t m_chan;
        channel_t m_chanPend;
        panId_t m_PANId;
        panId_t m_PANIdPend;
        bool m_EDA;
        bool m_EDAPend;
        bool m_CE;
        bool m_CEPend;
        uint16_t m_sourceAddress;
        uint16_t m_sourceAddressPend;
        uint32_t m_snLow;
        uint32_t m_snHigh;
        uint8_t  m_retries;
        uint8_t  m_retriesPend;
        uint8_t  m_randomDelaySlots;
        uint8_t  m_randomDelaySlotsPend;
        XBeeApiMACMode_e   m_macMode;
        XBeeApiMACMode_e   m_macModePend;
        XBeeApiDioConfig_e m_d[ XBEE_API_DIO_CHANNEL_COUNT ];
        XBeeApiDioConfig_e m_dPend[ XBEE_API_DIO_CHANNEL_COUNT ];
        uint8_t m_dioChangeDetectMask;
        uint8_t m_dioChangeDetectMaskPend;
        uint8_t m_dioLevels;
        uint8_t m_dioLevelsPend;
        uint16_t m_sampleRate;
        uint16_t m_sampleRatePend;
        uint32_t m_destHigh;
        uint32_t m_destHighPend;
        uint32_t m_destLow;
        uint32_t m_destLowPend;
        uint32_t m_writeCount;
        uint32_t m_applyCount;
        uint32_t m_resetCount; 
        uint32_t m_sampleCount;
        uint32_t m_restoreCount;
        bool     m_encryptionEnabled;
        bool     m_encryptionEnabledPend;
        
        time_t   m_ioDigitalUpdatedTime[ XBEE_API_DIO_CHANNEL_COUNT ];
        bool     m_ioDigitalState[ XBEE_API_DIO_CHANNEL_COUNT ];
        time_t   m_ioAnalogueUpdatedTime[ XBEE_API_ADC_CHANNEL_COUNT ];
        uint16_t m_ioAnalogueVal[ XBEE_API_ADC_CHANNEL_COUNT ];


        void SendCmd_uint8_t( const uint8_t        p_frameId,
                              const uint8_t* const p_data,
                              const uint8_t&       p_val );
        void SendCmd_uint8_t( const uint8_t        p_frameId,
                              const uint8_t* const p_data,
                              const uint8_t*       p_val );
        void SendCmd_uint16_t( const uint8_t        p_frameId,
                               const uint8_t* const p_data,
                               const uint16_t&       p_val );
        void SendCmd_uint32_t( const uint8_t        p_frameId,
                               const uint8_t* const p_data,
                               const uint32_t&       p_val );
                               
        virtual void SendCmd( const uint8_t p_frameId,
                              const uint8_t* const p_data,
                              const uint8_t* const p_val,
                              const uint8_t        p_len );
                      
        virtual size_t getResponseStatusPos( void ) const;


        /* Implement XBeeApiCmdDecoder interface */
        virtual bool decodeCallback( const uint8_t* const p_data, size_t p_len );

        virtual bool processResponseFrame( const uint8_t* const p_data, size_t p_len );

        virtual bool processIOFrame( const uint8_t* const p_data, size_t p_len, const size_t p_start );

        void resetCachedData( void );

    public:

        /** Constructor 
        
            \param p_device XBee device with which this object should be associated */
        XBeeApiCmdAt( XBeeDevice* const p_device = NULL );
       
        /** Destructor */
        virtual ~XBeeApiCmdAt( void ) {};
       
        time_t getDigitalState( const uint8_t p_chanNo, bool& p_state );
        time_t getAnalogueValue( const uint8_t p_chanNo, uint16_t& p_val );
       
        /** Request the hardware version identifier from the XBee.
            As the data is retrieved asynchronously to this call,
            once the response is received it can be accessed via
            getHardwareVersion() */
        bool requestHardwareVersion( void );
        bool requestFirmwareVersion( void );
        bool requestChannel( void );
        bool requestCoordinatorEnabled( void );
        bool requestEndDeviceAssociationEnabled( void );
        bool requestPanId( void );
        bool requestSourceAddress( void );
        bool requestSerialNumber( void );
        bool requestSerialNumberHigh( void );
        bool requestSerialNumberLow( void );
        bool requestDioConfig( const uint8_t p_chanNo );
        bool requestDioChangeDetectMask( void );
        bool requestSampleRate( void );
        
        /** Request the number of retries the XBee is configured to make.
            Note that the 802.15.4 MAC already allows 3 retries, so this parameter
            specifies an additional multiple of 3 retries
            As the data is retrieved asynchronously to this call,
            once the response is received it can be accessed via
            getRetries() 
        */
        bool requestRetries( void );
        
        /** Request the minimum value of the back-off exponent in the CSMA-CA algorithm
            used in collision avoidance.
        */
        bool requestRandomDelaySlots( void );
        
        bool requestMacMode( void );
        bool requestDestinationAddress( void );
        bool requestDestinationAddressHigh( void );
        bool requestDestinationAddressLow( void );
        
        bool requestWriteSettings( void );
        bool requestApplyChanges( void );
        bool requestReset( void );
        bool requestForceSample( void );
        bool requestRestoreDefaults( void );
        bool requestEncryptionEnabled( void );

        /** Read the XBee's hardware version identifier.  
       
            This method does not initiate any communication with the
            XBee - the identifier must previously have been requested 
            via requestHardwareVersion().  The method is non-blocking. */
        virtual bool getHardwareVersion( uint16_t* const p_ver );

        /** Read the XBee's firmware version identifier.  
       
            This method does not initiate any communication with the
            XBee - the identifier must previously have been requested 
            via requestFirmwareVersion().  The method is non-blocking. */
        virtual bool getFirmwareVersion( uint16_t* const p_ver );

        virtual bool getSerialNumber( uint64_t* const p_sn );
       
        virtual bool getChannel( uint8_t* const p_chan );
        virtual bool setChannel( uint8_t const p_chan );
       
        virtual bool getCoordinatorEnabled( bool* const p_en );
        virtual bool setCoordinatorEnabled( const bool p_en );
       
        virtual bool getEndDeviceAssociationEnabled( bool* const p_en ); 
        virtual bool setEndDeviceAssociationEnabled( const bool p_en  );
       
        virtual bool getPanId( panId_t* const p_id );       
        virtual bool setPanId( const panId_t p_id );       

        virtual bool getSourceAddress( uint16_t* const p_addr );       
        virtual bool setSourceAddress( const uint16_t p_addr );       

        virtual bool getRetries( uint8_t* const p_addr );       
        virtual bool setRetries( const uint8_t p_addr );       

        virtual bool getRandomDelaySlots( uint8_t* const p_addr );       
        virtual bool setRandomDelaySlots( const uint8_t p_addr );       

        virtual bool getMacMode( XBeeApiMACMode_e* const p_mode );       
        virtual bool setMacMode( const XBeeApiMACMode_e p_mode );  
        
        virtual bool getDioConfig( const uint8_t p_chanNo, XBeeApiDioConfig_e* const p_conf );
        virtual bool setDioConfig( const uint8_t p_chanNo, const XBeeApiDioConfig_e p_conf );
        
        virtual bool getDioChangeDetectMask( uint8_t* const p_mask );
        virtual bool setDioChangeDetectMask( const uint8_t p_mask );
        
        /** Note that this method will only return a value in the case that the levels have
            been previously set via the setDioLevels method and that the XBee has 
            returned a successful response.  There is no ability to simply read what the
            current digital outputs are set to */
        virtual bool getDioLevels( uint8_t* const p_mask );
        virtual bool setDioLevels( const uint8_t p_mask );

        /* TODO: Add protection to prevent p_interval being too small - see data sheet */
        virtual bool getSampleRate( uint16_t* const p_interval );
        virtual bool setSampleRate( const uint16_t p_interval );

        virtual bool getDestinationAddress( uint64_t* const p_address );
        virtual bool setDestinationAddress( const uint64_t p_address );

        virtual bool getDestinationAddressHigh( uint32_t* const p_address );
        virtual bool setDestinationAddressHigh( const uint32_t p_address );

        virtual bool getDestinationAddressLow( uint32_t* const p_address );
        virtual bool setDestinationAddressLow( const uint32_t p_address );

        /** Set the encryption key used by the XBee.  This key will only
            be used if encryption is enabled (see setEncryptionEnable ).
            Once set the key cannot be read back.
            
            \param p_key Pointer to a 16-byte array, with the first array
                         entry containing the most significant 8 bits of the
                         128-bit key
        */
        virtual bool setEncryptionKey( const uint8_t* p_key );
        
        /** Determine whether or not a call to setEncryptionKey has successfully
            set the key in the XBee.  Note that this will not query the key stored
            in the XBee and will only evern return true if a call to setEncryptionKey
            was made
            
            \returns true in the case that the XBee has responded to confirm that
                          a call to setEncryptionKey was successful
        */
        virtual bool getEncryptionKeySet( void );
        
        virtual bool getEncryptionEnabled( bool* const p_enabled );
        virtual bool setEncryptionEnabled( const bool  p_enabled );
};

#define XBEE_CMD_MAX_PARAM_LENGTH (16U)

#endif