/**
   @file
   @brief Support routines for writing very simple test harnesses

   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#if !defined TESTSUPPORTLITE_HPP
#define      TESTSUPPORTLITE_HPP

#include <cstring> // for size_t

/** Boiler plate test support for the TST_EQ, TST_TRUE, etc macros 
    \param _p Indicate whether or not the test passed
    \param _t Description of the test
*/
#define TST_BOILER( _p, _t) do { TEST_OUT( _p, _t ); if( !_p ) { FAIL_OUT(); } } while ( 0 )

/** Testing macro - expects _x and _q to be equal (using the equality operator)
    \param _x Value to be compared
    \param _y Value to be compared
    \param _t Description of test
*/
#define TST_EQ( _x, _y, _t ) do { bool passed = ((_x)==(_y)); TST_BOILER( passed, _t ); } while( 0 )

/** Testing macro - expects _x to be true
    \param _x Value to be evaluated
    \param _t Description of test
*/
#define TST_TRUE( _x, _t ) do { bool passed = (_x); TST_BOILER( passed, _t ); } while( 0 )

/** Testing macro - expects _x to be false
    \param _x Value to be evaluated
    \param _t Description of test
*/
#define TST_FALSE( _x, _t ) TST_TRUE( !( _x ), _t )

/** Tasks to be carried out when the tests are over */
#define TST_DONE() do { DONE_OUT(); } while( 0 )

/** Currently a very simple class just to encapsulate a few test statistics

    Implemented as a class rather than a set of global variables to simplify
    multiple instatitation
*/
class TestResults
{
    protected:
        /** Number of tests that have passed */
        std::size_t m_passed;
        /** Total number of tests run so far */
        std::size_t m_total;
    public:
        /* Default initialiser */
        TestResults( void );
        
        /* Register that a test has been run
           \parap p_passed Whether or not the test passed successfully */
        void registerResult( const bool p_passed );
        
        /* Query the total number of tests run so far */
        std::size_t getCount( void ) const;
        
        /* Query the number of tests passed so far */
        std::size_t getPassed( void ) const;
};

#endif // !defined TESTSUPPORTLITE_HPP