/**
   @file
   @brief MUTT controller, version 1.
          MUTT will keep going until an obstacle is detected at which
            point it'll stop and politely wait (indefinately) for the
            obstacle to move.
            
          See http://mbed.org/users/johnb/notebook/mutt-mbed-enabled-robot-vehicle/
   
   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#include "mbed.h"
#include "ArduinoMotorShield.hpp"

ArduinoMotorShield shield;
AnalogIn irFeedback(PTC2  /* A4 */);

/** Determine if the IR sensor is detecting an obstacle */
bool irSensesObstacle( void )
{
    /* IR sensor is TTL, so threshold is arbitary */
    static const float ir_thresh = 0.5;
    
    /* If analogue feedback is less than the threshold then there's an obstacle */
    return( irFeedback < ir_thresh );
}

/* Remove the comment to get some output on the serial line, however this
   may result in problems if the serial line's not connected (i.e. MUTT's
   not USB connected and is free roaming) */
#define SERIAL_OUTPUT

#if defined SERIAL_OUTPUT
#define OUTPUT_PERIOD 1000
Serial pc(USBTX, USBRX);
uint16_t ticker = 0;
uint16_t next_output = 0;

void serial_status( void )
{
    if( next_output == OUTPUT_PERIOD ) 
    {
        pc.printf( "MUTT: %04u Obstacle:%u  Current A: %0.2f   Current B: %0.2f\r\n",
                                            ticker++,irSensesObstacle(),
                                            shield.GetMotorCurrent(ArduinoMotorShield::MOTOR_A),
                                            shield.GetMotorCurrent(ArduinoMotorShield::MOTOR_B) );
        next_output = 0;
    }
    next_output++;
} 
#endif

int main( void )
{
    /* Set directions on the motors.  What you need here will depend on how 
       you've wired the motors to the motor shield.  If you find that one or 
       both wheels are turning backwards then you can either change the wiring 
       or change these settings */    
    shield.SetMotorPolarity( ArduinoMotorShield::MOTOR_A, ArduinoMotorShield::MOTOR_FORWARD );
    shield.SetMotorPolarity( ArduinoMotorShield::MOTOR_B, ArduinoMotorShield::MOTOR_BACKWARD );
    
    /* Forever is a long time ... */
    while( 1 )
    { 
        /* Determine if there's an obstacle */
        bool obstacle = irSensesObstacle();

#if defined SERIAL_OUTPUT
        serial_status();
#endif        
        
        /* Set the both motors to 1 (i.e. full power) in the case that there's
           no obstacle, set it to 0 (i.e. no power) in the case that an obstacle is
           detected */ 
        shield.SetMotorPower( ArduinoMotorShield::MOTOR_A, !obstacle );
        shield.SetMotorPower( ArduinoMotorShield::MOTOR_B, !obstacle );

        /* Wait 1 millisecond before goind round the loop again */
        wait_ms( 1 );
    }
}