/**
   @file
   @brief Class to abstract the interface to an Arduino Motor Shield R3
          See http://arduino.cc/en/Main/ArduinoMotorShieldR3
   
   @author John Bailey 

   @copyright Copyright 2014 John Bailey

   @section LICENSE
   
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

*/

#if !defined ARDUINOMOTORSHIELD_HPP
#define      ARDUINOMOTORSHIELD_HPP

#include "mbed.h"
#include <stdint.h>

/** Class to abstract the interface to an Arduino Motor Shield R3
    See http://arduino.cc/en/Main/ArduinoMotorShieldR3
    
    This class is a pretty thin wrapper around the analog inputs and 
    digital outputs which interface to the motor drives
    
    Currently supports:
        TARGET_KL46Z - Assumes the presence of the cross-wiring on PWM B 
                       (see "Issues" at http://mbed.org/users/johnb/notebook/mutt-mbed-enabled-robot-vehicle/ )
    
    @todo Support for bipolar stepper motor
    @todo Support for other targets
*/
class ArduinoMotorShield
{    
    public:
    
        typedef enum
        {
            MOTOR_FORWARD  = 0,
            MOTOR_BACKWARD = 1
        } MotorDirection_e;
    
        /** Enumeration to distinguish between the motors connected to the shield */    
        typedef enum
        {
            /* Motor 'A' */
            MOTOR_A = 0,
            /* Motor 'A' */
            MOTOR_B = 1
        } Motor_e;
    
    protected:
        /** Maximum number of motors which can be connected to the shield */
        static const uint8_t MOTOR_DRIVE_COUNT = 2U;
        
        /** Maximum current (per motor drive) supported by the shield */
        static const uint8_t MOTOR_MAX_CURRENT = 2U;
    
        PwmOut     m_a_motorControl;
        PwmOut     m_b_motorControl;
        DigitalOut m_a_brake;
        DigitalOut m_b_brake;
        AnalogIn   m_a_motorCurrent;
        AnalogIn   m_b_motorCurrent;
        DigitalOut m_a_motorDirection;    
        DigitalOut m_b_motorDirection;    
    
        PwmOut     *m_motorControl[ MOTOR_DRIVE_COUNT ];
        DigitalOut *m_brake[ MOTOR_DRIVE_COUNT ];
        AnalogIn   *m_motorCurrent[ MOTOR_DRIVE_COUNT ];
        DigitalOut *m_motorDirection[ MOTOR_DRIVE_COUNT ];
        
        float      m_speed[ MOTOR_DRIVE_COUNT ];
        bool       m_motorForward[ MOTOR_DRIVE_COUNT ];
    public:
    
        /** Constructor */
        ArduinoMotorShield();
        
        /** Set the motor power (and direction) for the specified motor
         
            \param p_motor Use this parameter to specify which motor to set the power on
            \param p_speed Set the power of the motor, ranging from -1 (full power, reverse direction) to
                           +1 (full power, forward direction)
        */
        void SetMotorPower( const Motor_e p_motor, const float p_speed, bool p_force = false );
        
        /** Retrieve the current being consumed by the specified motor
            \param p_motor The motor to retrieve the current for
        */
        float GetMotorCurrent( const Motor_e p_motor );
        
        /** Enable or disable the brake on the specified motor.  In the case that the braking
            option is enabled then (when appropriate) the shield will actively brake the motor
            rather than allowing it to coast down
            
            \param p_motor The motor to set the brake option for
            \param p_enable Specify whether braking should be enabled or disabled */
        void SetBrake( const Motor_e p_motor, bool p_enable );
        
        /** Set the virtual 'polarity' of a motor.  This allows you to swap the meaning of "forward"
            with respect to the SetMotorPower() method.  For example:
            
                shield.SetMotorPolarity( MOTOR_A, MOTOR_FORWARD );
                shield.SetMotorPower( MOTOR_A, 1.0F );
                // This call will cause the motor to change direction !
                shield.SetMotorPolarity( MOTOR_A, MOTOR_BACKWARD );
                // No further change is caused by this call
                shield.SetMotorPower( MOTOR_A, 1.0F );
                
            \param p_motor The motor to change the direction setting for
            \param p_dir   The direction which "forward" should map to.
        */
        void SetMotorPolarity( const Motor_e p_motor, const MotorDirection_e p_dir );
};

#endif // !defined ARDUINOMOTORSHIELD_HPP