/*
 *  This VL53L1CB Expansion board test application performs range measurements
 *  using the onboard embedded centre sensor, in multizone, polling mode.
 *  Measured ranges are ouput on the Serial Port, running at 115200 baud.
 *
 *
 * This is designed to work with MBed v2, & MBedOS v5 / v6.
 *
 *
 *  The Reset button can be used to restart the program.
 *
 * *** Note : 
 * Default Mbed build system settings disable print floating-point support.
 * Offline builds can enable this, again.
 * https://github.com/ARMmbed/mbed-os/blob/master/platform/source/minimal-printf/README.md
 * .\mbed-os\platform\mbed_lib.json
 *
 */
 
#include <stdio.h>
#include <time.h>

#include "mbed.h"
#include "XNucleo53L1A2.h"
#include "ToF_I2C.h"


// define the i2c comms pins
#define I2C_SDA   D14 
#define I2C_SCL   D15 



static XNucleo53L1A2 *board=NULL;
#if (MBED_VERSION  > 60300) 
    UnbufferedSerial  pc(USBTX, USBRX);
    extern "C" void wait_ms(int ms);
#else
    Serial pc(SERIAL_TX, SERIAL_RX); 
#endif

void process_interrupt( VL53L1 * sensor,VL53L1_DEV dev );
void print_results( int devSpiNumber, VL53L1_MultiRangingData_t *pMultiRangingData );

VL53L1_Dev_t                   devCentre;
VL53L1_Dev_t                   devLeft;
VL53L1_Dev_t                   devRight;
VL53L1_DEV                     Dev = &devCentre;

 
/*=================================== Main ==================================
=============================================================================*/
int main()
{   
    int status;
    VL53L1 * Sensor;
    uint16_t wordData;


    pc.baud(115200);  // baud rate is important as printf statements take a lot of time
    printf("Polling single multizone mbed = %d \r\n",MBED_VERSION);

// create i2c interface
    ToF_DevI2C *dev_I2C = new ToF_DevI2C(I2C_SDA, I2C_SCL);
    
    /* creates the 53L1A1 expansion board singleton obj */
   board = XNucleo53L1A2::instance(dev_I2C, A2, D8, D2);
    
    printf("board created!\r\n");

    /* init the 53L1A1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\r\n");
        return 0;
    }
       
        
    printf("board initiated! - %d\r\n", status);
  // create the sensor controller classes                                              

    VL53L1_RoiConfig_t roiConfig;
    
    roiConfig.NumberOfRoi =3;
    
    roiConfig.UserRois[0].TopLeftX = 0;
    roiConfig.UserRois[0].TopLeftY = 9; 
    roiConfig.UserRois[0].BotRightX = 4;
    roiConfig.UserRois[0].BotRightY = 5; 
    
    roiConfig.UserRois[1].TopLeftX = 5;
    roiConfig.UserRois[1].TopLeftY = 9; 
    roiConfig.UserRois[1].BotRightX = 9;
    roiConfig.UserRois[1].BotRightY = 4; 
    
    roiConfig.UserRois[2].TopLeftX = 11;
    roiConfig.UserRois[2].TopLeftY = 9; 
    roiConfig.UserRois[2].BotRightX = 15;
    roiConfig.UserRois[2].BotRightY = 5; 
    
// start of setup of central sensor
    Dev=&devCentre;
    
// configure the sensors
    Dev->comms_speed_khz = 400;
    Dev->comms_type = 1;
    Dev->i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
    devCentre.i2c_slave_address = NEW_SENSOR_CENTRE_ADDRESS;
    
    Sensor=board->sensor_centre;

    printf("configuring centre channel \n");
    
//    Sensor->VL53L1_RdWord(Dev, 0x01, &wordData);
//    printf("VL53L1X: %02X   %d\n\r", wordData,Dev->i2c_slave_address);
/* Device Initialization and setting */  
    status = Sensor->vl53L1_DataInit();
    status = Sensor->vl53L1_StaticInit();
    status = Sensor->vl53L1_SetPresetMode( VL53L1_PRESETMODE_MULTIZONES_SCANNING);
    status = Sensor->vl53L1_SetDistanceMode( VL53L1_DISTANCEMODE_LONG);
    
    
    status = Sensor->vl53L1_SetROI( &roiConfig);
    
    printf("VL53L1_SetROI %d \n",status);

    status = board->sensor_centre->vl53L1_StartMeasurement();  
        

// start of setup of left satellite
    Dev=&devLeft;
    
// configure the sensors
    Dev->comms_speed_khz = 400;
    Dev->comms_type = 1;
    Dev->i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
    devLeft.i2c_slave_address = NEW_SENSOR_LEFT_ADDRESS;
    
    Sensor=board->sensor_left;

    printf("configuring centre channel \n");
    
//    Sensor->VL53L1_RdWord(Dev, 0x01, &wordData);
//    printf("VL53L1X: %02X   %d\n\r", wordData,Dev->i2c_slave_address);
/* Device Initialization and setting */  
    status = Sensor->vl53L1_DataInit();
    status = Sensor->vl53L1_StaticInit();
    status = Sensor->vl53L1_SetPresetMode( VL53L1_PRESETMODE_MULTIZONES_SCANNING);
    status = Sensor->vl53L1_SetDistanceMode( VL53L1_DISTANCEMODE_LONG);
    
    
    status = Sensor->vl53L1_SetROI( &roiConfig);
    
    printf("VL53L1_SetROI %d \n",status);

    status = board->sensor_left->vl53L1_StartMeasurement();  
        

    VL53L1_MultiRangingData_t MultiRangingData;
    VL53L1_MultiRangingData_t *pMultiRangingData = &MultiRangingData;    
    
    // looping polling for results
    while (1)
    {
        pMultiRangingData = &MultiRangingData;    
        
        // wait for result
        status = board->sensor_centre->vl53L1_WaitMeasurementDataReady();  
        // get the result
        status = board->sensor_centre->vl53L1_GetMultiRangingData( pMultiRangingData);
        // if valid, print it
        if(status==0) {
            print_results(devCentre.i2c_slave_address, pMultiRangingData );
            status = board->sensor_centre->vl53L1_ClearInterruptAndStartMeasurement();

        } //if(status==0) 
        else
        {
            printf("VL53L1_GetMultiRangingData centre %d \n",status);
        }
        
        // wait for result
        status = board->sensor_left->vl53L1_WaitMeasurementDataReady();  
        // get the result
        status = board->sensor_left->vl53L1_GetMultiRangingData( pMultiRangingData);
        // if valid, print it
        if(status==0) {
            print_results(devLeft.i2c_slave_address, pMultiRangingData );
            status = board->sensor_left->vl53L1_ClearInterruptAndStartMeasurement();

        } //if(status==0) 
        else
        {
            printf("VL53L1_GetMultiRangingData centre %d \n",status);
        }
        
    }
}
    
    
    
 // prints the range result seen above   
void print_results( int devSpiNumber, VL53L1_MultiRangingData_t *pMultiRangingData )
{
            int no_of_object_found=pMultiRangingData->NumberOfObjectsFound;
            
            int RoiNumber=pMultiRangingData->RoiNumber;
//            int RoiStatus=pMultiRangingData->RoiStatus;

            if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0)) 
            {
                
  //              printf("MZI Count=%5d, ", pMultiRangingData->StreamCount);
   //             printf("RoiNumber%1d, ", RoiNumber);
   //               printf("RoiStatus=%1d, \n", RoiStatus);
                for(int j=0;j<no_of_object_found;j++){
                    if ((pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID) || 
                        (pMultiRangingData->RangeData[j].RangeStatus == VL53L1_RANGESTATUS_RANGE_VALID_NO_WRAP_CHECK_FAIL))
                    {
                        printf("*****************\t i2cAddr=0x%x \t RoiNumber=%d   \t status=%d, \t D=%5dmm, \t Signal=%2.2f Mcps, \t Ambient=%2.2f Mcps \n",
                        devSpiNumber,
                        RoiNumber,
                        pMultiRangingData->RangeData[j].RangeStatus,
                        pMultiRangingData->RangeData[j].RangeMilliMeter,
                        pMultiRangingData->RangeData[j].SignalRateRtnMegaCps/65536.0,
                        pMultiRangingData->RangeData[j].AmbientRateRtnMegaCps/65536.0);
                    }
                    else
                    {
//                               printf("RangeStatus %d %d\n",j, pMultiRangingData->RangeData[j].RangeStatus);
                    }
                }
            } // if (( no_of_object_found < 10 ) &&  ( no_of_object_found != 0)) 
            else
            {
//                       printf("no_of_object_found %d \n",no_of_object_found);
            }

}   

        
#if (MBED_VERSION  > 60300)
extern "C" void wait_ms(int ms)
 {
    thread_sleep_for(ms);
 }
 #endif 
     
  