#include "mbed.h"
#include "x_nucleo_53l0a1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/* This VL53L0X Expansion board test application performs a range measurement in polling mode
   on the onboard embedded top sensor. 
   The measured data is displayed on the on-board 4-digit display.

   User Blue button stops the current measurement and the entire program, releasing all resources.
   Reset button is used to restart the program. */

/* Polling operating modes don`t require callback function that handles IRQ 
   Callback IRQ functions are used only for measure that require interrupt */

/* GetMeasurement is asynchronous! It returns NOT_READY if the measurement value 
   is not ready to be read from the corresponding register. So you need to wait
   for the result to be ready */

#define VL53L0_I2C_SDA   D14 
#define VL53L0_I2C_SCL   D15 

#define RANGE   0

static X_NUCLEO_53L0A1 *board=NULL;
VL53L0X_RangingMeasurementData_t data_sensor_centre;
OperatingMode operating_mode;
    
/* flags that handle interrupt request */
bool int_sensor_centre=false, int_stop_measure=false;   

/* ISR callback function of the sensor_centre */
void SensorTopIRQ(void)
{
   int_sensor_centre=true;
   board->sensor_centre->DisableInterruptMeasureDetectionIRQ();
}   

/* ISR callback function of the user blue button to stop program */
void StopMeasureIRQ(void)
{
   int_stop_measure=true;
}

/* On board 4 digit local display refresh */
void DisplayRefresh(OperatingMode op_mode)
{   
   char str[5];
   
   if (op_mode==range_single_shot_polling || op_mode==range_continuous_interrupt || op_mode==range_continuous_polling)
   {
      if (data_sensor_centre.RangeStatus == 0) // we have a valid range.
      {
         sprintf(str,"%d",data_sensor_centre.RangeMilliMeter);
      }
      else
      {
         sprintf(str,"%s","----");
      }
   }
     
   board->display->DisplayString(str);
}

void RangeMeasure(DevI2C *device_i2c) {
   int status;

   /* creates the 53L0A1 expansion board singleton obj */
   board=X_NUCLEO_53L0A1::Instance(device_i2c, A2, D8, D2);
    
   board->display->DisplayString("53L0");
   /* init the 53L0A1 expansion board with default values */
   status=board->InitBoard();
   if(status)
      printf("Failed to init board!\n\r");   
   operating_mode=range_single_shot_polling;
   /* start the measure on sensor top */
   status=board->sensor_centre->StartMeasurement(operating_mode, SensorTopIRQ);
   if(!status)
   {
      while(1)
      {
         status=board->sensor_centre->GetMeasurement(operating_mode, &data_sensor_centre);
         DisplayRefresh(operating_mode);
         if(int_stop_measure) // Blue Button isr was triggered
         {
            status=board->sensor_centre->StopMeasurement(operating_mode); // stop the measure and exit
            int_stop_measure = false;
            printf("\nProgram stopped!\n\n\r");
            break;
         }
      }
   }
   board->display->DisplayString("BYE");
   delete board;        
}    

/*=================================== Main ==================================
 Press the blue user button to stop the measurements in progress    
=============================================================================*/
int main()
{   
#if USER_BUTTON==PC_13  // we are cross compiling for Nucleo-f401 
   InterruptIn stop_button (USER_BUTTON);
   stop_button.rise (&StopMeasureIRQ);  
#endif   
   DevI2C *device_i2c =new DevI2C(VL53L0_I2C_SDA, VL53L0_I2C_SCL);     
        
   RangeMeasure(device_i2c);  // start continuous measures
}

