/*
  @file rtos_project.cpp
  @author Joe Roop
  @date 4/25/2014

Assignment: RTOS Project
Objective:
            Utilize the mbed as a remote sensing device. The device
            will communicate over Ehternet to a server and that 
            server will display the data in real-time and allow 
            sever clients to connect. The use case for the project is
            to provide a community the ability to use the sensor to 
            check the traffic of their local laundry facility. The device
            can be connected to any router with DHCP and will send its
            data out over a web socket to be collected

 @par Copyright:
  Copyright (c) 2014 Joe Roop
 @par License:
  This software is distributed under the terms of the GNU Lesser General Public License
*/       

#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "Websocket.h"
#include "C12832_lcd.h" 


typedef enum {CONNECT = 1, DISCONNECT} state_t;
//Threads
void thread_eth(void const *args);
void thread_ws(void const *args);
void thread_send(void const *args);

Thread *proxy_eth;
Thread *proxy_ws;
Thread *proxy_send;

EthernetInterface eth;
Websocket ws("ws://washwatch.duckdns.org:8080/");


//lcd
C12832_LCD  lcd;
AnalogIn sonar(p17); //hook up gnd, 3.3 red, p17 yellow
Mutex lcd_mutex;

void prnt(char msg[40],bool clear,int line);
void connect(void);
int cnt;

int main()
{    
    Thread t1(thread_eth);    
    Thread t2(thread_ws);
    Thread t3(thread_send);
    
    proxy_eth = &t1;
    proxy_ws = &t2;
    proxy_send = &t3;
    
    proxy_eth->signal_set(CONNECT); //start the system
    while(1){
         Thread::wait(1000);   
    }
}

//Threads
//keep track of ethernet connection
void thread_eth(void const *args){
    osEvent evt;
    int32_t sig, cnt, ret;
    eth.init();
    while(1){
        evt = Thread::signal_wait(0); //will time out then loop not needed
        sig = evt.value.signals;
        switch(sig){
            case CONNECT:
                cnt = 0; //make a saw tooth
                while((ret = eth.connect()) != 0){
                    lcd_mutex.lock();
                    lcd.locate(0,0);
                    lcd.printf("eth try...%d, %d",++cnt, ret);
                    lcd_mutex.unlock();
                    cnt = cnt > 1000 ? 0 : cnt; //limit the cnt
                    Thread::wait(250);
                } //loop forever trying to connect 
                proxy_ws->signal_set(CONNECT);
                break;    
        }
    }
}
//keep track of ws connection
void thread_ws(void const *args){
    osEvent evt;
    int32_t sig, ret, cnt;
    while(1){
        evt = Thread::signal_wait(0); //will time out then loop not needed
        sig = evt.value.signals;
        switch(sig){
            case CONNECT:
                cnt = 50;
                ws.close(); //close the connection because send was not working
                while((ret = ws.connect()) != 0 && cnt-- > 0){
                    lcd_mutex.lock();
                    lcd.locate(0,10);
                    lcd.printf("ws try connect...%d",cnt);
                    lcd_mutex.unlock();
                    Thread::wait(250);
                } //loop x times trying to connect
                if(ret != 0) proxy_eth->signal_set(CONNECT); //couldn't connect 
                else proxy_send->signal_set(CONNECT); //got a ws connection
                break;    
        }
    }
    
}
//send data at intervals
void thread_send(void const *args){
    osEvent evt;
    int32_t sig, ret, cnt;
    char msg[100];
    float range;
    while(1){
        evt = Thread::signal_wait(0); //will time out then loop not needed
        sig = evt.value.signals;
        switch(sig){
            case CONNECT:
                cnt = 0; //make this a saw tooth
                do{
                    range = sonar*3.3/0.0032*0.0328084;
                    sprintf(msg, "{\"msg\":\"sonar\",\"range\":%3.2f}", range);
                    ret = ws.send(msg);
                    lcd_mutex.lock();
                    lcd.locate(0,20);
                    lcd.printf("%d range: %-3.2f ft %d ",ret, range, ws.is_connected());
                    lcd_mutex.unlock();
                    Thread::wait(100); //10Hz
                }while(ret != -1);
                proxy_ws->signal_set(CONNECT); //lost connection to ws so try ws
                break;    
        }
    }
}